package com.agilex.healthcare.mobilehealthplatform.datalayer.vital;

import java.util.Collection;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.*;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.*;
import com.agilex.healthcare.utility.NullChecker;

public class FetchVitalEntriesRequestHandler extends AbstractFetchPatientDataListRequestHandler<VitalDataLayer, VitalEntries, VitalEntry>{
	
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory
			.getLog(FetchVitalEntriesRequestHandler.class);
	

	@Override
	protected VitalEntries invokeDataLayerToFetchList(VitalDataLayer datalayer, RequestMessage requestMessage, PatientDataFetchRequestReader messageReader) {
		PatientIdentifiers translatedIdentifiers = translatePatientIdentifier(requestMessage);
		
		VitalEntries vitalEntries = new VitalEntries();
		if(NullChecker.isNullish(translatedIdentifiers)){
			return vitalEntries;
		} else {
            for(PatientIdentifier patientIdentifier : translatedIdentifiers) {
                VitalEntries retrieved = datalayer.fetchVitalEntries(patientIdentifier, getFilter(requestMessage));
                fixIdentifiers(retrieved, patientIdentifier);
                vitalEntries.addAll(retrieved);
            }
		}

		return vitalEntries;
	}
	
	private VitalEntryFilter getFilter(RequestMessage request) {
		return request.getValue("vitals-filter");
	}
	
	private ScopeFilter getScopeFilter(RequestMessage request) {
		return request.getValue("scope-filter");
	}
	
	private String getDomain(RequestMessage request) {
		return request.getValueAsString("domain");
	}
	
	private void fixIdentifiers(VitalEntries vitalEntries, PatientIdentifier patientIdentifier) {
		for(VitalEntry entry : vitalEntries) {
			entry.setPatientIdentifier(patientIdentifier);
		}
	}

	private Collection<DataSystem> determineDataSystemList(RequestMessage requestMessage) {
		DomainServiceRegistry registry = new DomainServiceRegistry();
		return registry.getDataSystems(getScopeFilter(requestMessage), getDomain(requestMessage));
	}
	
	private PatientIdentifiers translatePatientIdentifier(RequestMessage requestMessage) {
		PatientIdentifier patientIdentifier = requestMessage.getValue("patient-identifier");
		
		Collection<DataSystem> dataSystems = determineDataSystemList(requestMessage);
		
		for (DataSystem targetDataSystem : dataSystems) {
			
			String targetAssigningAuthority = targetDataSystem.getPatientAssigningAuthority();
			
			if (targetDataSystem.getSystemIdentifier().equalsIgnoreCase("vista")) {
				MhpUser currentUser = getCurrentUser();
				targetAssigningAuthority = String.format(targetDataSystem.getPatientAssigningAuthority(),currentUser.getVistaLocation()); 
			}
            PatientIdentifiers patientIdentifiers = translatePatientIdentifier(patientIdentifier, targetAssigningAuthority);

            if(NullChecker.isNotNullish(patientIdentifiers))
                return patientIdentifiers;

        }

        return null;
	}
	
	private MhpUser getCurrentUser(){
		return MhpUserFactory.createFromSecurityContext();
	}
	
	private PatientIdentifiers translatePatientIdentifier(PatientIdentifier patientIdentifier, String targetAssigningAuthority) {
		
		PatientIdentifiers matcingPatientIdentifiers = new PatientIdentifiers();

		if(targetAssigningAuthority.equals(patientIdentifier.getAssigningAuthority())){
			logger.debug("No translations are neccessary as patient's assigning authority and target assigning authority are same for the patient identifier ");
			matcingPatientIdentifiers.add(patientIdentifier);
			return matcingPatientIdentifiers;
		}
		
		// Get all the identifiers for given patient identifier 
		PatientIdentifiers correspondingIdentifiers = MhpObjectFactory.getInstance().getPatientCorrelationService()
				.getCorrespondIdentifiers(patientIdentifier);
		
		
		// Get all the identifiers matching the target assigning authority
		for (PatientIdentifier possibleMatch : correspondingIdentifiers) {
			if (possibleMatch.getAssigningAuthority().toLowerCase().contains(targetAssigningAuthority.toLowerCase())) {
				matcingPatientIdentifiers.add(possibleMatch);
			}
		}
		
		logger.debug("translated patient identifier for assigning authority into corresponding identifiers");
		
		return matcingPatientIdentifiers;
	}
	
	
}
