package com.agilex.healthcare.mobilehealthplatform.datalayer.securemessage;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageCode;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageContacts;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageFolder;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageFolders;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageUser;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessages;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.utils.PropertyHelper;
import com.agilex.healthcare.utility.NullSafeStringComparer;

import edu.emory.mathcs.backport.java.util.Collections;

public class SecureMessageDataService {

	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(SecureMessageDataService.class);
	private Router router;
	private PropertyHelper propertyHelper;
	private static final String HA_MODE = "ha.mode";

	public SecureMessageDataService(PropertyHelper propertyHelper) {
		this.propertyHelper = propertyHelper;
		this.router = new Router();
	}

	public SecureMessageDataService(Router router, PropertyHelper propertyHelper) {
		this.router = router;
		this.propertyHelper = propertyHelper;
	}

	public SecureMessageUser fetchSecureMessageUser(PatientIdentifier userIdentifier) {
		RequestMessage request = SecureMessageRequestBuilder.forSecureMessageUser().forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessageUser secureMessageUser = SecureMessageUserResponseReader.fromResponse(responseMessage).getUser();

        SecureMessageUserVerificationCache dataService = SecureMessageUserVerificationCache.getInstance();
        dataService.saveUserId(userIdentifier, secureMessageUser.getPatientId());

        return secureMessageUser;
	}

	public SecureMessageFolders fetchFolders(PatientIdentifier userIdentifier) {

		RequestMessage request = SecureMessageRequestBuilder.forSecureMessageFolders().forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessageFolders folders = SecureMessageFolderResponseReader.fromResponse(responseMessage).getFolders();
		folders.updatePatientIdentifier(userIdentifier);
		folders = filterProviderFolders(folders);

		return folders;
	}

	public SecureMessageContacts fetchContacts(PatientIdentifier userIdentifier) {
		RequestMessage request = SecureMessageRequestBuilder.forSecureMessageContacts().forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessageContacts contacts = SecureMessageContactResponseReader.fromResponse(responseMessage).getContacts();
		contacts.updatePatientIdentifier(userIdentifier);

		return contacts;
	}

	public SecureMessages fetchSecureMessages(PatientIdentifier userIdentifier, SecureMessageFilter filter) {
		RequestMessage request = SecureMessageRequestBuilder.forSecureMessageData().forFilter(filter).forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessages secureMessages = SecureMessageResponseReader.fromResponse(responseMessage).getSecureMessages();
		
		Collections.sort(secureMessages);
		
		setPaginationInformation(secureMessages, filter);

		return secureMessages;
	}

	public SecureMessage readSecureMessage(PatientIdentifier userIdentifier, SecureMessage secureMessage) {
		logger.debug("reading a SecureMessage");

		RequestMessage request = SecureMessageRequestBuilder.forReadSecureMessage().forSecureMessage(secureMessage).forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessage message = SecureMessageResponseReader.fromResponse(responseMessage).getSecureMessage();

		return message;
	}

	public SecureMessage sendSecureMessage(PatientIdentifier userIdentifier, SecureMessage secureMessage) {
		logger.debug("sending a SecureMessage");

		RequestMessage request = SecureMessageRequestBuilder.forSendSecureMessage().forSecureMessage(secureMessage).forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessage message = SecureMessageResponseReader.fromResponse(responseMessage).getSecureMessage();

		return message;
	}

	public SecureMessage replySecureMessage(PatientIdentifier userIdentifier, SecureMessage secureMessage) {
		logger.debug("replying to a SecureMessage");

		RequestMessage request = SecureMessageRequestBuilder.forReplySecureMessage().forSecureMessage(secureMessage).forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessage message = SecureMessageResponseReader.fromResponse(responseMessage).getSecureMessage();

		return message;
	}

	public SecureMessage moveSecureMessage(PatientIdentifier userIdentifier, String messageId, String newFolderId) {
		logger.debug("moving a SecureMessage");

		SecureMessage messageContainer = new SecureMessage();
		messageContainer.setUniqueId(messageId);
		messageContainer.setFolderId(newFolderId);

		RequestMessage request = SecureMessageRequestBuilder.forMoveSecureMessage().forSecureMessage(messageContainer).forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessage message = SecureMessageResponseReader.fromResponse(responseMessage).getSecureMessage();

		return message;
	}

	public SecureMessage saveSecureMessageDraft(PatientIdentifier userIdentifier, SecureMessage secureMessage) {
		logger.debug("saving a SecureMessage as a draft");

		RequestMessage request = SecureMessageRequestBuilder.forSaveSecureMessage().forSecureMessage(secureMessage).forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessage message = SecureMessageResponseReader.fromResponse(responseMessage).getSecureMessage();

		return message;
	}

	public SecureMessage sendSecureMessageDraft(PatientIdentifier userIdentifier, SecureMessage secureMessage) {
		logger.debug("sending a SecureMessage draft");

		RequestMessage request = SecureMessageRequestBuilder.forSendSecureMessageDraft().forSecureMessage(secureMessage).forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		ResponseMessage responseMessage = router.execute(request);
		SecureMessage message = SecureMessageResponseReader.fromResponse(responseMessage).getSecureMessage();

		return message;
	}

	public void deleteSecureMessageDraft(PatientIdentifier userIdentifier, String messageId) {
		logger.debug("deleting a SecureMessage");

		SecureMessage messageContainer = new SecureMessage();
		messageContainer.setPatientIdentifier(userIdentifier);
		messageContainer.setUniqueId(messageId);

		RequestMessage request = SecureMessageRequestBuilder.forDeleteSecureMessageDraft().forSecureMessage(messageContainer).forUserIdentifier(userIdentifier).forDomain(Domain.secureMessage).build();
		router.execute(request);
	}

	SecureMessageFolders filterProviderFolders(SecureMessageFolders retrievedFolders) {
		String haMode = propertyHelper.getProperty(HA_MODE);
		if (!NullSafeStringComparer.areEqual("veteran", haMode)) {
			return retrievedFolders;
		}

		SecureMessageFolders filteredFolders = new SecureMessageFolders();

		for (SecureMessageFolder folder : retrievedFolders) {
			if (SecureMessageCode.FOLDER_ID_INBOX.toString().equals(folder.getUniqueId()) || SecureMessageCode.FOLDER_ID_DRAFT.toString().equals(folder.getUniqueId()) || SecureMessageCode.FOLDER_ID_DELETED.toString().equals(folder.getUniqueId()) || SecureMessageCode.FOLDER_ID_SENT.toString().equals(folder.getUniqueId()) || Integer.parseInt(folder.getUniqueId()) >= 0) {
				filteredFolders.add(folder);
			}
		}

		return filteredFolders;
	}
	
	void setPaginationInformation(SecureMessages secureMessages, SecureMessageFilter filter) {
		secureMessages.setPageNumber(Integer.parseInt(filter.getPageNumber()));
		secureMessages.setPageSize(Integer.parseInt(filter.getPageSize()));
		if(secureMessages.size() == Integer.parseInt(filter.getPageSize()))
			secureMessages.setHasNext(true);
	}
	
}
