package com.agilex.healthcare.mobilehealthplatform.datalayer.patient;

import java.util.Collections;
import java.util.regex.Pattern;

import com.agilex.healthcare.mobilehealthplatform.domain.Address;
import com.agilex.healthcare.mobilehealthplatform.domain.NextOfKin;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.mobilehealthplatform.domain.formatters.AddressDispayFormatter;
import com.agilex.healthcare.utility.NullChecker;

public class PatientFormatter {
	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientFormatter.class);
	private final Pattern commaWithNoSpacePattern = Pattern.compile("[a-z],[a-z]", Pattern.CASE_INSENSITIVE);

	public void sort(Patients patients) {
		Collections.sort(patients);
	}

	public Patients formatPatients(Patients patients) {
		if (patients != null)
			for (Patient patient : patients) {
				formatPatient(patient);
			}
		return patients;
	}

	public void formatPatient(Patient patient) {
		if (patient != null) {
			formatMiddleInitial(patient);
			formatSsnForDisplay(patient);
			formatDisplayName(patient);
			formatGender(patient);
			fillInAge(patient);
		}
	}

	public void formatDemographics(PatientDemographics demographics) {
		if (demographics != null) {
			formatAddress(demographics);
			formatNextOfKin(demographics);
		}
	}

	private void formatNextOfKin(PatientDemographics demographics) {
		NextOfKin nok = demographics.getNextOfKin();
		if (nok != null) {
			if (NullChecker.isNullish(nok.getName()) && NullChecker.isNullish(nok.getPhoneNumber()) && NullChecker.isNullish(nok.getRelationship()))
				demographics.setNextOfKin(null);
		}
	}

	private void formatAddress(PatientDemographics demographics) {
		Address address = demographics.getAddress();
		if ((address != null) && NullChecker.isNullish(address.getFormattedDisplayAddress1())) {
			AddressDispayFormatter formatter = new AddressDispayFormatter();
			formatter.createFormatDisplay(address);
		}
		if ((address != null) && NullChecker.isNullish(address.getFormattedDisplayAddress1())) {
			demographics.setAddress(null);
		}
	}

	private void formatGender(Patient patient) {
		if (NullChecker.isNotNullish(patient.getGender())) {
			if (patient.getGender().toLowerCase().contentEquals("m"))
				patient.setGender("MALE");
			if (patient.getGender().toLowerCase().contentEquals("f"))
				patient.setGender("FEMALE");
		}
	}

	private void formatDisplayName(Patient patient) {

		String displayName = patient.getDisplayName();
		if (NullChecker.isNullish(displayName)) {
			displayName = appendNameToDisplayName(displayName, patient.getLastName(), null);
			displayName = appendNameToDisplayName(displayName, patient.getFirstName(), ", ");
			displayName = appendNameToDisplayName(displayName, patient.getMiddleInitial(), " ");
		} else if (commaWithNoSpacePattern.matcher(displayName).find()) {
			// format "last,first" -> "last, first"
			displayName = patient.getDisplayName().replace(",", ", ");
		}
		patient.setDisplayName(displayName);

	}

	private String appendNameToDisplayName(String existingName, String nameToAppend, String seperator) {
		if (NullChecker.isNotNullish(nameToAppend)) {
			if (NullChecker.isNotNullish(existingName)) {
				existingName = existingName + seperator + nameToAppend;
			} else {
				existingName = nameToAppend;
			}
		}
		return existingName;
	}

	private void formatMiddleInitial(Patient patient) {
		if (NullChecker.isNotNullish(patient.getMiddleName()))
			patient.setMiddleInitial(patient.getMiddleName().substring(0, 1));
	}

	private void formatSsnForDisplay(Patient patient) {
		String ssn = patient.getSsn();
		if (NullChecker.isNotNullish(ssn) && ssn.length() == 9) {
			ssn = ssn.substring(0, 3) + "-" + ssn.substring(3, 5) + "-" + ssn.substring(5, 9);
			patient.setSsn(ssn);
		}
	}

	private void fillInAge(Patient patient) {
		if (NullChecker.isNullish(patient.getAge())) {
			logger.debug("calculating age for patient");

			if (patient.getDateOfBirth() != null){
				long numericAge;
				numericAge = AgeCalculator.calculateAge(patient.getDateOfBirth().getDate());
				if (NullChecker.isNotNullish(numericAge)) {
					String age = Long.toString(numericAge);
					patient.setAge(age);
				}
			}
			
		} else {
			logger.debug("no need to calculate age for patient");
		}
	}
}
