package com.agilex.healthcare.mobilehealthplatform.datalayer.patient;

import java.util.Collection;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestHandler;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestReader;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DataSystem;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DomainServiceRegistry;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.mobilehealthplatform.utils.DefaultAssigningAuthorityUpdater;

public class PatientFetchRequestHandler implements RequestHandler {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientFetchRequestHandler.class);

	@Override
	public ResponseMessage handle(RequestMessage requestMessage) {
		PatientDataLayer datalayer = getDataLayer(requestMessage);

		PatientDataFetchRequestReader messageReader = PatientDataFetchRequestReader.fromRequest(requestMessage);

		PatientIdentifiers translatedPatientIdentifiers = getTranslatedPatientIdentifiers(messageReader);
		PatientIdentifier translatedPatientIdentifier = translatedPatientIdentifiers.get(0);

		Patient patient = datalayer.getPatient(translatedPatientIdentifier);

		fixPatientIdentifier(requestMessage, patient);

		PatientFormatter formatter = new PatientFormatter();
		formatter.formatPatient(patient);

		return PatientFetchResponseBuilder.fromPatientData(patient).build();
	}

	private PatientDataLayer getDataLayer(RequestMessage requestMessage) {
		DomainServiceRegistry serviceRegistry = new DomainServiceRegistry();
		return serviceRegistry.getDataLayerByMessage(requestMessage);
	}

	private void fixPatientIdentifier(RequestMessage requestMessage, Patient patient) {
		DomainServiceRegistry serviceRegistry = new DomainServiceRegistry();
		DataSystem system = serviceRegistry.getSystemByMessage(requestMessage);

		String targetAssigningAuthority = system.getPatientAssigningAuthority();
		if ("vista".equalsIgnoreCase(system.getSystemIdentifier())) {
			targetAssigningAuthority = String.format(targetAssigningAuthority, getCurrentUser().getVistaLocation());
		}

		DefaultAssigningAuthorityUpdater.fillDefaultAssigningAuthority(patient, targetAssigningAuthority);
	}

	private MhpUser getCurrentUser() {
		return MhpUserFactory.createFromSecurityContext();
	}

	private PatientIdentifiers getTranslatedPatientIdentifiers(PatientDataFetchRequestReader messageReader) {
		PatientIdentifiers translatedPatientIdentifiers = null;

		DomainServiceRegistry registry = new DomainServiceRegistry();
		Collection<DataSystem> datasystemList = registry.getDataSystems(messageReader.getScopeFilter(), messageReader.getDomain());

		DataSystem dataSystem = null;

		// we will assume that we will only fetch from one system
		for (DataSystem tempdataSystem : datasystemList) {
			dataSystem = tempdataSystem;
			break;
		}
		
		if(dataSystem == null) {
			logger.error("!! System is not configured properly - review the ServiceRegistry to ensure that a system/assigning-authority is setup");
			return new PatientIdentifiers();
		}

		String targetSystemIdentifier = dataSystem.getSystemIdentifier();
		String targetAssigningAuthority = dataSystem.getPatientAssigningAuthority();

		if ("vista".equalsIgnoreCase(targetSystemIdentifier)) {
			MhpUser currentUser = getCurrentUser();
			targetAssigningAuthority = String.format(dataSystem.getPatientAssigningAuthority(), currentUser.getVistaLocation());
		}

		if (!targetAssigningAuthority.contentEquals(messageReader.getPatientIdentifier().getAssigningAuthority())) {
			translatedPatientIdentifiers = translatePatientIdentifier(messageReader.getPatientIdentifier(), targetAssigningAuthority);
		} else {
			translatedPatientIdentifiers = new PatientIdentifiers();
			translatedPatientIdentifiers.add(messageReader.getPatientIdentifier());
		}

		return translatedPatientIdentifiers;
	}

	private PatientIdentifiers translatePatientIdentifier(PatientIdentifier patientIdentifier, String targetAssigningAuthority) {
		PatientIdentifiers correspondingPatientIdentifiers = MhpObjectFactory.getInstance().getPatientCorrelationService().getCorrespondIdentifiers(patientIdentifier, targetAssigningAuthority);
		logTranslatedPatientIdentifiers(patientIdentifier, targetAssigningAuthority, correspondingPatientIdentifiers);
		return correspondingPatientIdentifiers;
	}

	private void logTranslatedPatientIdentifiers(PatientIdentifier originalPatientIdentifier, String targetAssigningAuthority, PatientIdentifiers correspondIdentifiers) {
		StringBuilder builder = new StringBuilder();
		for (PatientIdentifier correspondingIdentifier : correspondIdentifiers) {
			builder.append("[" + correspondingIdentifier + "]");
		}
		
		logger.debug("Translated patient identifier ");
	}

}
