package com.agilex.healthcare.mobilehealthplatform.datalayer.patient;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.patientsearchstringparser.SearchStringParser;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.NullChecker;

public class PatientDataService {
	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientDataService.class);

	private Router router;

	public PatientDataService() {
		this(new Router());
	}

	public PatientDataService(Router router) {
		this.router = router;
	}

	public Patient getPatient(PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
		RequestMessage request = PatientFetchRequestBuilder.forRetrieve().forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).build();
		ResponseMessage responseMessage = router.execute(request);

		return PatientFetchResponseReader.fromResponse(responseMessage).getPatient();
	}

	public PatientIdentifiers fetchPatientIdentifiers(PatientIdentifier patientIdentifier) {
		return MhpObjectFactory.getInstance().getPatientCorrelationService().getCorrespondIdentifiers(patientIdentifier);
	}

	public PatientDemographics getPatientDemographics(PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
//		logger.info(String.format("retrieve patient demographics by id [patientid=%s]", patientIdentifier));

		if (scopeFilter.isLongitudinal()) {
			RequestMessage request = PatientFetchRequestBuilder.forRetrieveDemographics().forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).build();
			ResponseMessage responseMessage = router.execute(request);

			return PatientFetchResponseReader.fromResponse(responseMessage).getDemographics();
		} else {
			RequestMessage request = PatientFetchRequestBuilder.forNextOfKin().forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).build();
			ResponseMessage responseMessage = router.execute(request);
			
			NextOfKin nextOfKin = PatientFetchResponseReader.fromResponse(responseMessage).getNextOfKin();
			
			request = PatientFetchRequestBuilder.forContactInfo().forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).build();
			responseMessage = router.execute(request);
			
			PatientDemographics patientDemographics = PatientFetchResponseReader.fromResponse(responseMessage).getContactInfo();
			
			if(patientDemographics == null) 
				patientDemographics = new PatientDemographics();
				
			
			patientDemographics.setPatientIdentifier(patientIdentifier);
			patientDemographics.setNextOfKin(nextOfKin);
			
			return patientDemographics;
		}
	}

	public PatientSearchResults patientSearch(Patient prototype, ScopeFilter scopeFilter) {
		return patientSearch(prototype, null, scopeFilter);
	}

	public PatientSearchResults patientSearch(Patient prototype, PatientSearchPaging pagingInfo, ScopeFilter scopeFilter) {
		logger.info("performing patient search by prototype");
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		criteria.setPrototype(prototype);
		return patientSearch(criteria, pagingInfo, scopeFilter);
	}

	public PatientSearchResults patientSearch(String searchString, String siteCode, ScopeFilter scopeFilter) {
		return patientSearch(searchString, siteCode, null, scopeFilter);
	}

	public PatientSearchResults patientSearch(String searchString, String vistaSiteCode, PatientSearchPaging pagingInfo, ScopeFilter scopeFilter) {
//		logger.info("performing patient search by searchstring [" + searchString + "]");
		PatientSearchCriteria criteria = SearchStringParser.parseSearchString(searchString);
		if (NullChecker.isNullish(criteria.getSearchString())) {
			criteria.setSearchString(searchString);
		}
		criteria.setVistaSiteCode(vistaSiteCode);
		return patientSearch(criteria, pagingInfo, scopeFilter);
	}

	public PatientSearchResults patientSearch(PatientSearchCriteria criteria, ScopeFilter scopeFilter) {
		return patientSearch(criteria, null, scopeFilter);
	}

	public PatientSearchResults patientSearch(PatientSearchCriteria criteria, PatientSearchPaging pagingInfo, ScopeFilter scopeFilter) {
		logger.info("performing patient search");

		RequestMessage request = PatientSearchRequestBuilder.forSearch().forCriteria(criteria).forScopeFilter(scopeFilter).build();
		ResponseMessage response = router.execute(request);
		Patients patients = PatientSearchResponseReader.fromResponse(response).getPatients();

		PatientSearchResults searchResults = enforceMaxRecords(patients, pagingInfo);

		return searchResults;
	}

	private PatientSearchResults enforceMaxRecords(Patients patients, PatientSearchPaging pagingInfo) {
		Paging paging = new Paging();
		return paging.enforceMaxRecords(patients, pagingInfo);
	}

}
