package com.agilex.healthcare.mobilehealthplatform.datalayer.patient;

import java.util.Collection;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestHandler;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestReader;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DataSystem;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DomainServiceRegistry;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.NullChecker;

public class FetchDemographicsRequestHandler implements RequestHandler {

	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(FetchDemographicsRequestHandler.class);

	@Override
	public ResponseMessage handle(RequestMessage requestMessage) {
		PatientSupplementalDemographicsDataLayer datalayer = getDataLayer(requestMessage);

		PatientDataFetchRequestReader messageReader = PatientDataFetchRequestReader.fromRequest(requestMessage);

		PatientIdentifiers translatedIdentifiers = translatePatientIdentifiers(messageReader);

		PatientDemographics demographics = null;

		if (NullChecker.isNullish(translatedIdentifiers)) {
			logger.debug("Corresponding patient identifier does not exist, so returning empty demographics");
			demographics = new PatientDemographics();
		} else {
            for(PatientIdentifier identifier : translatedIdentifiers) {
			    demographics = datalayer.fetchDemographics(identifier);
            }
		}

		if (demographics != null) {
			demographics.setPatientIdentifier(messageReader.getPatientIdentifier());
			PatientFormatter formatter = new PatientFormatter();
			formatter.formatDemographics(demographics);
		}

		return PatientFetchResponseBuilder.fromDemographics(demographics).build();
	}

	private PatientSupplementalDemographicsDataLayer getDataLayer(RequestMessage requestMessage) {
		DomainServiceRegistry serviceRegistry = new DomainServiceRegistry();
		return serviceRegistry.getDataLayerByMessage(requestMessage);
	}

	private Collection<DataSystem> determineDataSystemList(ScopeFilter scopeFilter) {
		DomainServiceRegistry registry = new DomainServiceRegistry();
		return registry.getDataSystems(ScopeFilter.getInstanceForLongitudinalScope(), Domain.supplementalDemographics);
	}

	private PatientIdentifiers translatePatientIdentifiers(PatientDataFetchRequestReader requestReader) {
		// if there are multiple patient id's in vista for a given patient, we will pick the first one
		Collection<DataSystem> dataSystems = determineDataSystemList(requestReader.getScopeFilter());

        // TEST
		for (DataSystem targetDataSystem : dataSystems) {
			if (targetDataSystem.getPatientAssigningAuthority().equalsIgnoreCase(requestReader.getPatientIdentifier().getAssigningAuthority())) {
                PatientIdentifiers patientIdentifiers = new PatientIdentifiers();
                patientIdentifiers.add(requestReader.getPatientIdentifier());
				return patientIdentifiers;
			} else {
				return MhpObjectFactory.getInstance().getPatientCorrelationService().getCorrespondIdentifiers(requestReader.getPatientIdentifier(), targetDataSystem.getPatientAssigningAuthority());

			}
		}

		return null;
	}

}
