package com.agilex.healthcare.mobilehealthplatform.datalayer.paingoal;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports.HtmlReportBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.ReportGenerator;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoal;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoalReport;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoals;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PainGoalEntryStatus;

public class PainGoalReportGenerator {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ReportGenerator.class);

	private PainGoals painGoals;
	
	private PainGoals activeGoals;
	private PainGoals completedGoals;
	private PainGoals deletedGoals;
	
	public PainGoalReportGenerator(PainGoals paingoals) {
		this.painGoals = paingoals;
	}

	public PainGoalReport generateReport() {
		PainGoalReport report = new PainGoalReport();

		report.setName("Pain Goal Report");

		report.setReport(configureReportValue());


		return report;
	}

	private String configureReportValue() {

        String header = "\n";
		header += HtmlReportBuilder.create("Pain Goal Report").center().bold().nl().build() +
                HtmlReportBuilder.create("Created on " + DateHelper.formatDateTimeInReportFormat(DateHelper.getToday())).center().bold().nl().build() ;


        String reportText = header;

        sortPainGoals();

        reportText += createActiveGoalsSection();
        reportText += createCompletedGoaslSection();
        reportText += createDeletedGoalsSection();

        logger.debug(reportText);
		return reportText;
	}



	private void sortPainGoals() {
		activeGoals = new PainGoals();
		completedGoals = new PainGoals();
		deletedGoals = new PainGoals();

        if (null != painGoals) {
            for (PainGoal goal : painGoals) {

                switch (goal.getStatus()) {
                case PainGoalStatusActive:
                    activeGoals.add(goal);
                    break;
                case PainGoalStatusComplete:
                    completedGoals.add(goal);
                    break;
                case PainGoalStatusDeleted:
                    deletedGoals.add(goal);
                    break;
                default:
                    break;
                }
            }
        }
    }

	private String createActiveGoalsSection() {

		String goalsSection = HtmlReportBuilder.create("Active Goals").underline().bold().nl().build();

		for (PainGoal goal : activeGoals) {
			goalsSection += buildActiveGoalsSection(goal);
            createCompletedActionStepsFragment(goal);
		}

		return HtmlReportBuilder.create(goalsSection).nl().nl().build();
	}

    private String createCompletedActionStepsFragment(PainGoal goal) {
        String completedActionSteps;
        completedActionSteps = HtmlReportBuilder.create("Completed Action Steps:").nl().build();

        String goalEntries = "";
        if (NullChecker.isNotNullish(goal.getPainGoalEntries())) {
            for (PainGoalEntry entry : goal.getPainGoalEntries()) {
                if (entry.getStatus() == PainGoalEntryStatus.PainGoalEntryStatusComplete){
                    goalEntries += buildGoalEntrySection(entry);
                }
            }
        }
        completedActionSteps += HtmlReportBuilder.create(goalEntries).unorderedList().build();
        return completedActionSteps;
    }

    private String createCompletedGoaslSection() {
        String goalsSection = HtmlReportBuilder.create("Completed Goals").underline().bold().nl().build();

        for (PainGoal goal : completedGoals) {
            goalsSection += HtmlReportBuilder.create(buildCompletedGoalsSection(goal)).nl().build();
        }

        return HtmlReportBuilder.create(goalsSection).nl().nl().build();
    }

    private String createDeletedGoalsSection() {
        String goalsSection = HtmlReportBuilder.create("Deleted Goals").underline().bold().nl().build();

        for (PainGoal goal : deletedGoals) {
            goalsSection += HtmlReportBuilder.create(buildDeletedGoalsSection(goal)).nl().build();
        }

        return goalsSection;
    }

	private String buildGoalEntrySection(PainGoalEntry entry) {
		String section = DateHelper.formatDate(entry.getEntryDate()) + ": " + entry.getComments();
		return HtmlReportBuilder.create(section).listItem().build();
	}

	private String buildActiveGoalsSection(PainGoal goal) {

        String name = createGoalNameframent(goal);
        String startedDate = createDateStartedFragment(goal);
        String targetDate = createTargetDateFragment(goal);
        String percentComplete = createPercentCompleteFragment(goal);
        String completedActionStep = createCompletedActionStepsFragment(goal);
        String section = name +  percentComplete + startedDate + targetDate + completedActionStep;

		return section;
	}

    private String buildCompletedGoalsSection(PainGoal goal) {
        String name = createGoalNameframent(goal);
        String startedDate = createDateStartedFragment(goal);
        String completedDate = createCompletedDateFragment(goal);
        String percentComplete = createPercentCompleteFragment(goal);
        String details = createDetailsFragment(goal);
        String section = name +  percentComplete + startedDate + completedDate + details;
        return section;
    }

    private String buildDeletedGoalsSection(PainGoal goal) {
        String name = createGoalNameframent(goal);
        String startedDeletedDate = createDateStartedFragment(goal) + createDateDeletedFragment(goal);
        String section = name +  startedDeletedDate;
        return section;
    }

    private String createDetailsFragment(PainGoal goal) {
        return HtmlReportBuilder.create("Details: " + goal.getDetails()).nl().build();
    }

    private String createCompletedDateFragment(PainGoal goal) {
        return HtmlReportBuilder.create("Completed Date: " + dateString(goal.getDateCompleted())).nl().build();
    }

//    private String createDateStartedAndDeletedFragment(PainGoal goal) {
//        return HtmlReportBuilder.create("Date Started: " + dateString(goal.getDateStarted()) + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" + "Date Deleted: " + dateString(goal.getDateDeleted())).nl().build();
//    }

    private String createTargetDateFragment(PainGoal goal) {
        return HtmlReportBuilder.create("Target Date: " + dateString(goal.getTargetDate())).nl().build();
    }

    private String createDateStartedFragment(PainGoal goal) {
        return HtmlReportBuilder.create("Date Started: " + dateString(goal.getDateStarted())).nl().build();
    }

    private String createDateDeletedFragment(PainGoal goal) {
        return HtmlReportBuilder.create("Date Deleted: " + dateString(goal.getDateDeleted())).nl().build();
    }

    private String createPercentCompleteFragment(PainGoal goal) {
        int percentComplete = 0;
        if (NullChecker.isNotNullish(goal.getPercentComplete())){
            percentComplete = Integer.parseInt(goal.getPercentComplete());
        }
        String text =  HtmlReportBuilder.create("").progressBar(percentComplete).build();
        return text;
    }

    private String createGoalNameframent(PainGoal goal) {
        return HtmlReportBuilder.create("Name: " + goal.getName()).nl().build();
    }

	private String dateString(Date dateStarted) {

		if (NullChecker.isNotNullish(dateStarted)) {
			return DateHelper.formatDate(dateStarted);
		} else {
			return "Date Not Set";
		}

	}

}
