package com.agilex.healthcare.mobilehealthplatform.datalayer.paingoal;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import org.apache.log4j.Logger;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataEditRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataResponseReader;
import com.agilex.healthcare.mobilehealthplatform.datalayer.pain.PainDiaryEntryDataService;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PainGoalEntryStatus;

import edu.emory.mathcs.backport.java.util.Collections;

public class PainGoalDataService {
	public static final String RequestKeyForShouldIncludeDeletedGoals = "ShouldIncludeDeleted";
	public static final String RequestKeyForShouldIncludeDeletedGoalsTrue = "true";
	public static final String RequestKeyForShouldIncludeDeletedGoalsFalse = "false";
	public static final String RequestKeyForPainGoalID = "painGoalId";

    private static final  Logger logger = Logger.getLogger(PainDiaryEntryDataService.class);

	private Router router = new Router();

	public PainGoalDataService() {
	}
	
	public PainGoals fetchPainGoals(PatientIdentifier patientIdentifier) {
		logger.debug("fetching pain goals");
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.painGoal).forPatientIdentifier(patientIdentifier).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).setValue(RequestKeyForShouldIncludeDeletedGoals, RequestKeyForShouldIncludeDeletedGoalsFalse).build();
		ResponseMessage responseMessage = router.execute(request);
		PainGoals painGoals = PatientDataResponseReader.<PainGoals, PainGoal> fromResponse(responseMessage).getDataListNoNull(PainGoals.class);		
		
		sortPainGoals(painGoals);
		
		return painGoals;
	}

	public PainGoals fetchAllPainGoals(PatientIdentifier patientIdentifier) {
        return fetchAllPainGoals(patientIdentifier, DateFilterFactory.createEmptyFilter());
	}

    public PainGoals fetchAllPainGoals(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
        logger.debug("fetching all pain goals");
        RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.painGoal).forPatientIdentifier(patientIdentifier).forDateFilter(dateFilter).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).setValue(RequestKeyForShouldIncludeDeletedGoals, RequestKeyForShouldIncludeDeletedGoalsTrue).build();
        ResponseMessage responseMessage = router.execute(request);
        PainGoals painGoals = PatientDataResponseReader.<PainGoals, PainGoal> fromResponse(responseMessage).getDataListNoNull(PainGoals.class);

        sortPainGoals(painGoals);

        return painGoals;
    }

    public PainGoalEntries fetchAllPainGoalEntries(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier) {
		logger.debug("fetching all pain goal entries for an ID");
	
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.painGoalEntries).forPatientIdentifier(patientIdentifier).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).forDataIdentifier(dataIdentifier).build();
		ResponseMessage responseMessage = router.execute(request);
		PainGoalEntries painGoalEntries = PatientDataResponseReader.<PainGoalEntries, PainGoalEntry> fromResponse(responseMessage).getDataListNoNull(PainGoalEntries.class);		
		
		sortPainGoalEntries(painGoalEntries);
				
		return painGoalEntries;
	}

	private void sortPainGoalEntries(PainGoalEntries painGoalEntries) {
		if (NullChecker.isNotNullish(painGoalEntries)) {
			painGoalEntries.sortDescending("entryDate");
		}		
	}

	public PainGoal createNewPainGoal(PatientIdentifier patientIdentifier, PainGoal painGoal) {
		logger.debug("creating a new pain goal");

		RequestMessage request = PatientDataEditRequestBuilder.forCreate().forDomain(Domain.painGoal).forData(painGoal).forPatientIdentifier(patientIdentifier)
				.forScope(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);
		PainGoal createdPainGoal = PatientDataResponseReader.<PainGoals, PainGoal> fromResponse(responseMessage).getDataItem();

		return createdPainGoal;
	}
	
	public PainGoal updatePainGoal(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier, PainGoal painGoal) {
		logger.debug("updating a pain goal");

        addSurrogate(painGoal);

		RequestMessage request = PatientDataEditRequestBuilder.forUpdate().forDomain(Domain.painGoal).forData(painGoal).forDataIdentifier(dataIdentifier)
				.forPatientIdentifier(patientIdentifier).forScope(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);
		
		PainGoal updatedPainGoal = PatientDataResponseReader.<PainGoals, PainGoal> fromResponse(responseMessage).getDataItem();

		updatedPainGoal.setPatientIdentifier(patientIdentifier);
		updatedPainGoal.setDataIdentifier(dataIdentifier);
		
		return updatedPainGoal;
	}
	
	private void sortPainGoals(PainGoals painGoals) {
		Collections.sort(painGoals, new PainGoalsSortByStatusAndCreationDateComparator());
		
		for (PainGoal goal : painGoals) {
			sortPainGoalEntries(goal);
		}
	}

	private void sortPainGoalEntries(PainGoal goal) {
		if (NullChecker.isNotNullish(goal.getPainGoalEntries())) {
			goal.getPainGoalEntries().sortDescending("entryDate");
		}
	}

	public PainGoalEntry createNewPainGoalEntry(PatientIdentifier patientIdentifier, String goalId, PainGoalEntry painGoalEntry) {
		logger.debug("creating a new pain goal entry");

        addSurrogate(painGoalEntry);

		painGoalEntry.setEntryDate(DateHelper.getToday());
		if (painGoalEntry.getStatus() == null) {
			painGoalEntry.setStatus(PainGoalEntryStatus.PainGoalEntryStatusActive);
		}
		
		RequestMessage request = PatientDataEditRequestBuilder.forCreate().forDomain(Domain.painGoalEntries).forData(painGoalEntry).forPatientIdentifier(patientIdentifier)
				.forScope(ScopeFilter.getInstanceForLongitudinalScope()).setValue(RequestKeyForPainGoalID, goalId).build();
		ResponseMessage responseMessage = router.execute(request);
		PainGoalEntry createdPainGoalEntry = PatientDataResponseReader.<PainGoalEntries, PainGoalEntry> fromResponse(responseMessage).getDataItem();

		return createdPainGoalEntry;
	}

	public PainGoalEntry updatePainGoalEntry(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier, String goalId, PainGoalEntry painGoalEntry) {
		logger.debug("updating a pain goal entry");

        addSurrogate(painGoalEntry);

		RequestMessage request = PatientDataEditRequestBuilder.forUpdate().forDomain(Domain.painGoalEntries).forData(painGoalEntry).forDataIdentifier(dataIdentifier)
				.forPatientIdentifier(patientIdentifier).forScope(ScopeFilter.getInstanceForLongitudinalScope()).setValue(RequestKeyForPainGoalID, goalId).build();
		ResponseMessage responseMessage = router.execute(request);
		
		PainGoalEntry updatedPainGoalEntry = PatientDataResponseReader.<PainGoalEntries, PainGoalEntry> fromResponse(responseMessage).getDataItem();

		updatedPainGoalEntry.setPatientIdentifier(patientIdentifier);
		updatedPainGoalEntry.setDataIdentifier(dataIdentifier);
		
		return updatedPainGoalEntry;
	}

	public void deletePainGoalEntry(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier, String goalId, PainGoalEntry painGoalEntry) {
		logger.debug("deleting a pain goal entry");

        addSurrogate(painGoalEntry);

		RequestMessage request = PatientDataEditRequestBuilder.forDelete().forDomain(Domain.painGoalEntries).forData(painGoalEntry).forDataIdentifier(dataIdentifier)
				.forPatientIdentifier(patientIdentifier).forScope(ScopeFilter.getInstanceForLongitudinalScope()).setValue(RequestKeyForPainGoalID, goalId).build();
		ResponseMessage responseMessage = router.execute(request);
		
		PatientDataResponseReader.<PainGoalEntries, PainGoalEntry> fromResponse(responseMessage).getDataItem();

	}

	public PainGoal fetchPainGoal(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier) {
		logger.debug("fetching pain goal");
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveSingleById().forDomain(Domain.painGoal).forPatientIdentifier(patientIdentifier).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).forDataIdentifier(dataIdentifier).build();
		ResponseMessage responseMessage = router.execute(request);
		PainGoal painGoal = PatientDataResponseReader.<PainGoals, PainGoal> fromResponse(responseMessage).getDataItem();		
		
		sortPainGoalEntries(painGoal);
		
		return painGoal;
	}

	public void deletePainGoal(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier, PainGoal painGoal) {
		logger.debug("deleting a pain goal");

        addSurrogate(painGoal);

		RequestMessage request = PatientDataEditRequestBuilder.forDelete().forDomain(Domain.painGoal).forData(painGoal).forDataIdentifier(dataIdentifier)
				.forPatientIdentifier(patientIdentifier).forScope(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);
		
		PatientDataResponseReader.<PainGoals, PainGoal> fromResponse(responseMessage).getDataItem();
		
	}

    private void addSurrogate(PatientData data) {
        MhpUserFactory.addSurrogateToPatientData(data);
    }
}
