package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistrations;
import com.agilex.healthcare.mobilehealthplatform.domain.NotificationUser;
import com.agilex.healthcare.mobilehealthplatform.domain.NotificationUsers;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.mobilehealthplatform.patientcorrelation.PatientCorrelationService;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.NullChecker;

public class NotificationUserDataService {

	private PatientDataService patientDataService;
	private PatientCorrelationService patientCorrelationService;
	private DeviceRegistrationDataService deviceRegistrationDataService;

	public NotificationUserDataService() {
		patientDataService = new PatientDataService();
		deviceRegistrationDataService = new DeviceRegistrationDataService();
		patientCorrelationService = MhpObjectFactory.getInstance().getPatientCorrelationService();
	}

	public NotificationUserDataService(PatientDataService patientDataService, DeviceRegistrationDataService deviceRegistrationDataService, PatientCorrelationService patientCorrelationService) {
		this.patientDataService = patientDataService;
		this.deviceRegistrationDataService = deviceRegistrationDataService;
		this.patientCorrelationService = patientCorrelationService;
	}

	public NotificationUsers searchNotificationUsers(String searchString, String siteCode) {
		Patients patients = patientDataService.patientSearch(searchString, siteCode, ScopeFilter.getInstanceForLongitudinalScope());

		if (NullChecker.isNullish(patients)) {
			return new NotificationUsers();
		}

		Map<String, Patient> userIdsMap = new HashMap<String, Patient>();
		List<String> userIdList = new LinkedList<String>();
		for (Patient patient : patients) {
			PatientIdentifiers edipiIdentifiers = patientCorrelationService.getCorrespondIdentifiers(patient.getPatientIdentifier(), "EDIPI");
			//Excluding the patients if there is no EDIPI identifier
			if (NullChecker.isNotNullish(edipiIdentifiers)){
				String edipiIdentifier =  edipiIdentifiers.get(0).getUniqueId();
				userIdsMap.put(edipiIdentifier, patient);
				userIdList.add(edipiIdentifier);
			}
		}
		DeviceRegistrations deviceRegistrations = deviceRegistrationDataService.fetchRegistrations(userIdList);
		Set<NotificationUser> notificationUserSet = initializeNotificationUserSet(userIdsMap, deviceRegistrations);

		return convertSetToNotificationUsers(notificationUserSet);
	}

	private Set<NotificationUser> initializeNotificationUserSet(Map<String, Patient> userIdsMap, DeviceRegistrations deviceRegistrations) {
		Set<NotificationUser> notificationUserSet = new HashSet<NotificationUser>();
		for (DeviceRegistration deviceRegistration : deviceRegistrations) {
			String userId = deviceRegistration.getUserId();
			Patient patient = userIdsMap.get(userId);
			NotificationUser notificationUser = createNotificationUser(patient, userId);
			notificationUserSet.add(notificationUser);
		}
		return notificationUserSet;
	}

	private NotificationUser createNotificationUser(Patient patient, String edipiIdentifier) {
		NotificationUser notificationUser = new NotificationUser();
		notificationUser.setAge(patient.getAge());
		notificationUser.setDateOfBirth(patient.getDateOfBirth());
		notificationUser.setFirstName(patient.getFirstName());
		notificationUser.setMiddleName(patient.getMiddleName());
		notificationUser.setLastName(patient.getLastName());
		notificationUser.setGender(patient.getGender());
		notificationUser.setSsn(patient.getSsn());
		notificationUser.setUserId(edipiIdentifier);
		return notificationUser;
	}

	private NotificationUsers convertSetToNotificationUsers(Set<NotificationUser> notificationUserSet) {
		NotificationUser[] notificationUserArray = new NotificationUser[notificationUserSet.size()];
		notificationUserSet.toArray(notificationUserArray);
		NotificationUsers notificationUsers = new NotificationUsers(notificationUserArray);
		return notificationUsers;
	}
}
