package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response.Status;

import com.agilex.healthcare.mobilehealthplatform.datalayer.audit.AuditLogDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.AuditLog;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistrations;
import com.agilex.healthcare.mobilehealthplatform.domain.Notification;
import com.agilex.healthcare.mobilehealthplatform.domain.Notifications;
import com.agilex.healthcare.mobilehealthplatform.domain.Property;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AuditLogCode;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DataLayerFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ServiceRegistry;
import com.agilex.healthcare.mobilehealthplatform.utils.PropertyHelper;
import com.agilex.healthcare.utility.NullChecker;

public class NotificationDataService {
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(NotificationDataService.class);
	private static final int MAX_NUMBER_OF_DISPLAYED_NOTIFICATIONS = 50;
	private NotificationDataLayer notificationDataLayer;
	private NotificationCommunicationDataLayer notificationCommunicationDataLayer;
	private DeviceRegistrationDataService deviceRegistrationDataService;
	private AuditLogDataService auditLogDataService;
	private static final String NOTIFICATIONS_OPTOUT_ENABLED = "notifications.option_enabled";
	private PropertyHelper propertyHelper;

	public NotificationDataService(PropertyHelper propertyHelper) {
		this.propertyHelper = propertyHelper;
		notificationDataLayer = getDataLayerFactory().getNotificationDataLayer();
		notificationCommunicationDataLayer = getDataLayerFactory().getNotificationCommunicationDataLayer();
		deviceRegistrationDataService = new DeviceRegistrationDataService();
		auditLogDataService = new AuditLogDataService();
	}

	public NotificationDataService(NotificationDataLayer notificationDataLayer, 
			NotificationCommunicationDataLayer notificationSenderLayer, 
			DeviceRegistrationDataService deviceRegistrationDataService,
			AuditLogDataService auditLogDataService,
			PropertyHelper propertyHelper) {
		this.notificationDataLayer = notificationDataLayer;
		this.notificationCommunicationDataLayer = notificationSenderLayer;
		this.deviceRegistrationDataService = deviceRegistrationDataService;
		this.auditLogDataService = auditLogDataService;
		this.propertyHelper = propertyHelper;
	}

	public Notification fetchNotification(String userId, String notificationId) {
		recordAuditLog(userId, AuditLogCode.NOTIFICATION, AuditLogCode.NOTIFICATION_SUBTYPE_RETRIEVED);
		return this.notificationDataLayer.getNotification(userId, notificationId);
	}

	public Notifications fetchAllNotifications(String userId, DateFilter filter) {
		Notifications fetchedNotifications = this.notificationDataLayer.getActiveNotifications(userId, filter);
		recordFetchedNotifications(fetchedNotifications);
		fetchedNotifications.sortDescending("date");
		return truncateNotifications(fetchedNotifications);
	}

	public Property fetchNotificationStatus() {
		Boolean optInStatus = Boolean.valueOf(propertyHelper.getProperty(NOTIFICATIONS_OPTOUT_ENABLED));
		return new Property(optInStatus.toString());
	}

	private Notifications truncateNotifications(Notifications fetchedNotifications) {
		Notifications truncatedNotifications = new Notifications();
		if (fetchedNotifications.size() > MAX_NUMBER_OF_DISPLAYED_NOTIFICATIONS) {
			List<Notification> truncatedList = fetchedNotifications.subList(0, MAX_NUMBER_OF_DISPLAYED_NOTIFICATIONS);
			truncatedNotifications.addAll(truncatedList);
		} else {
			truncatedNotifications = fetchedNotifications;
		}
		return truncatedNotifications;
	}

	public Notification saveNotification(Notification notification) {
		validateNotification(notification);

		notification.setActiveFlag(true);
		if (notification.getDate() == null) {
			notification.setDate(new Date());
		}

		boolean tokensExist = sendNotification(notification);
		if (tokensExist) {
			return this.notificationDataLayer.saveNotification(notification);
		} else {
			return null; // will fail when tokens are validated
		}
	}

	public void deleteNotification(String userId, String notificationId) {
		recordAuditLog(userId, AuditLogCode.NOTIFICATION, AuditLogCode.NOTIFICATION_SUBTYPE_DELETED);
		Notification fetchedNotification = fetchNotification(userId, notificationId);
		this.notificationDataLayer.deleteNotification(fetchedNotification);
	}

	public void deleteNotification(Notification notification) {
		recordAuditLog(notification.getUserId(), AuditLogCode.NOTIFICATION, AuditLogCode.NOTIFICATION_SUBTYPE_DELETED);
		this.notificationDataLayer.deleteNotification(notification);
	}

	private void validateNotification(Notification notification) {
		if (NullChecker.isNullish(notification.getBody()) || NullChecker.isNullish(notification.getUserId())) {
			throw new WebApplicationException(Status.BAD_REQUEST);
		}
	}
	
	private boolean sendNotification(Notification notification) {
		DeviceRegistrations deviceRegistrations = this.deviceRegistrationDataService.fetchRegistrations(notification.getUserId());

		validateDeviceRegistrations(deviceRegistrations);

		List<String> deviceTokensToRemove = new LinkedList<String>();
		for (DeviceRegistration deviceRegistration : deviceRegistrations) {
			String personalizedMessage = generatePersonalizedMessage(notification, deviceRegistration);
			List<String> invalidDeviceTokens = notificationCommunicationDataLayer.send(deviceRegistration.getDeviceToken(), personalizedMessage);
			this.recordAuditLog(notification.getUserId(), AuditLogCode.NOTIFICATION, AuditLogCode.NOTIFICATION_SUBTYPE_SENT);
			deviceTokensToRemove.addAll(invalidDeviceTokens);
		}

		removeDeviceTokens(notification.getUserId(), deviceTokensToRemove);

		return NullChecker.isNotNullish(deviceRegistrations);
	}

	private String generatePersonalizedMessage(Notification notification, DeviceRegistration deviceRegistration) {
		String personalizedMessage;
		boolean deviceRegistrationHasNoFirstName = NullChecker.isNullish(deviceRegistration.getFirstName());

		if (deviceRegistrationHasNoFirstName) {
			LOGGER.warn("User is missing first name in Device Registration data store.  Skipping personalized notification generation.");
			personalizedMessage = notification.getBody();
		} else {
			personalizedMessage = String.format("%s - %s", deviceRegistration.getFirstName(), notification.getBody());
		}
		return personalizedMessage;

	}

	private void validateDeviceRegistrations(DeviceRegistrations deviceRegistrations) {
		if (deviceRegistrations.size() == 0) {
			throw new WebApplicationException(Status.PRECONDITION_FAILED);
		}
	}

	private void removeDeviceTokens(String userId, List<String> deviceTokens) {
		for (String deviceToken : deviceTokens) {
			this.recordAuditLog(userId, AuditLogCode.NOTIFICATION, AuditLogCode.NOTIFICATION_SUBTYPE_REJECTED);
			this.deviceRegistrationDataService.deleteRegistration(userId, deviceToken);
		}
	}

	private void recordAuditLog(String userId, String type, String subType) {
		AuditLog auditLog = new AuditLog(userId, type, subType, new Date(), null);
		auditLogDataService.saveAudit(auditLog);
	}

	private void recordFetchedNotifications(Notifications notifications) {
		for (Notification notification : notifications) {
			recordAuditLog(notification.getUserId(), AuditLogCode.NOTIFICATION, AuditLogCode.NOTIFICATION_SUBTYPE_RETRIEVED);
		}
	}

	private DataLayerFactory getDataLayerFactory() {
		return ServiceRegistry.getInstance().getPrimaryCommunity().getDataLayerFactory();
	}
}
