package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import javapns.Push;
import javapns.communication.exceptions.CommunicationException;
import javapns.communication.exceptions.KeystoreException;
import javapns.devices.Device;
import javapns.devices.exceptions.InvalidDeviceTokenFormatException;
import javapns.notification.PushNotificationPayload;
import javapns.notification.PushedNotification;
import javapns.notification.ResponsePacket;

import org.apache.log4j.Logger;
import org.json.JSONException;

import com.agilex.healthcare.mobilehealthplatform.utils.PropertyHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.StringTruncator;

/**
 * NotificationCommunicationDataLayerJavaPNS
 * Wrapper for JavaPNS library
 */
public class NotificationCommunicationDataLayerJavaPNS implements NotificationCommunicationDataLayer {

	private static final Logger LOGGER = Logger.getLogger(NotificationCommunicationDataLayerJavaPNS.class);
	private static final String NOTIFICATIONS_USE_PRODUCTION_APNS = "notifications.use_production_apns";
	private static final String NOTIFICATIONS_KEYSTORE_FILENAME = "notifications.keystore_filename";
	private static final String NOTIFICATIONS_KEYSTORE_PW = "notifications.keystore_password";

	private String keystoreFilename;
	private String keystorePassword;
	private boolean useProductionAPNS;

	public NotificationCommunicationDataLayerJavaPNS(PropertyHelper propertyHelper) {
		this.keystoreFilename = propertyHelper.getProperty(NOTIFICATIONS_KEYSTORE_FILENAME);
		this.keystorePassword = propertyHelper.getProperty(NOTIFICATIONS_KEYSTORE_PW);
		this.useProductionAPNS = Boolean.valueOf(propertyHelper.getProperty(NOTIFICATIONS_USE_PRODUCTION_APNS));
	}

	/** {@inheritDoc} */
	public List<String> send(String deviceToken, String message) {
		return send(new String[] { deviceToken }, message);
	}
	
	/** {@inheritDoc} */
	public List<String> send(String[] deviceTokens, String message) {
		if (NullChecker.isNullish(message)) {
			throw new IllegalArgumentException("message cannot be null or empty");
		}
		if (deviceTokens == null) {
			throw new NullPointerException("deviceTokens cannot be null");
		}
		String[] defensiveDeviceTokens = Arrays.copyOf(deviceTokens, deviceTokens.length);
		if (deviceTokens.length <= 0) {
			throw new IllegalArgumentException("deviceTokens must have at least one element");
		}
		
		InputStream keystore = null;
		try {
			keystore = NotificationCommunicationDataLayerJavaPNS.class.getClassLoader().getResourceAsStream(keystoreFilename);
			PushNotificationPayload payload = PushNotificationPayload.complex();
	        payload.addAlert(StringTruncator.truncate(message, MAX_NOTIFICATION_LENGTH_IN_BYTES));
			payload.addSound(DEFAULT_SOUND_FILE);
			
			int numberOfThreads = (deviceTokens.length/200)+1;
			List<PushedNotification> notificationResults = 
					Push.payload(
							payload, 
							keystore, 
							keystorePassword, 
							useProductionAPNS, 
							numberOfThreads,
							defensiveDeviceTokens);
			
			return getInvalidDeviceTokens(notificationResults);
		} catch (KeystoreException e) {
			LOGGER.error("Keystore Exception", e);
			throw new RuntimeException("Keystore Exception: Please check the keystore/certificates associated with the application and APNS server");
		} catch (CommunicationException e) {
			LOGGER.error("Communication Exception", e);
			throw new RuntimeException("Communication Exception: This could be an issue with connecting with Apple's APNS Server.");
		} catch (JSONException e) {
			LOGGER.error("JSON Exception while reading the APNS Notification payload", e);
			throw new RuntimeException("JSONException: The payload is likely malformed.");
		} catch (Exception e) {
			LOGGER.error("Unknown Exception while reading the APNS notification payload", e);
			throw new RuntimeException("Exception occurred while attempting to send a custom payload APNS message.");
		} finally {
			if (keystore != null) {
				try {
					keystore.close();
				} catch (IOException e) {
					LOGGER.error("IOException", e);
					throw new RuntimeException("Exception occurred while attempting to close keystore input stream.");
				}
			}
		}
	}

	/** {@inheritDoc} */
	private List<String> getInvalidDeviceTokens(List<PushedNotification> notifications) {
		List<String> invalidDeviceTokens = new LinkedList<String>();
		
		for (PushedNotification notification : notifications) {
			if (notification.isSuccessful() == false) {
				LOGGER.debug("Error sending message to token");
				
				ResponsePacket errorResponsePacket = notification.getResponse();
				if (errorResponsePacket != null || notification.getException() instanceof InvalidDeviceTokenFormatException) {
					// JUSTIFICATION: Only considering tokens that have an error response packet as invalid.  See <http://code.google.com/p/javapns/wiki/FeedbackService>
					LOGGER.debug("Removing token");
					invalidDeviceTokens.add(notification.getDevice().getToken());
				}
			}
		}
		
		return invalidDeviceTokens;
	}

	/** {@inheritDoc} */
	public List<String> fetchInvalidDeviceTokens() {
		List<String> invalidDeviceTokens = new LinkedList<String>();
		
		InputStream keystore = null;
		try {
			keystore = NotificationCommunicationDataLayerJavaPNS.class.getClassLoader().getResourceAsStream(keystoreFilename);
			List<Device> inactiveDevices = Push.feedback(keystore, keystorePassword, useProductionAPNS);
			
			for (Device inactiveDevice : inactiveDevices) {
				invalidDeviceTokens.add(inactiveDevice.getToken());
			}
		} catch (KeystoreException e) {
			LOGGER.error("Keystore Exception", e);
			throw new RuntimeException("Keystore Exception: Please check the keystore/certificates associated with the application and APNS server");
		} catch (CommunicationException e) {
			LOGGER.error("Communication Exception", e);
			throw new RuntimeException("Communication Exception: This could be an issue with connecting with Apple's APNS Server.");
		} finally {
			if (keystore != null) {
				try {
					keystore.close();
				} catch (IOException e) {
					LOGGER.error("IOException", e);
					throw new RuntimeException("Exception occurred while attempting to close keystore input stream.");
				}
			}
		}
		
		return invalidDeviceTokens;
	}

}
