package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import java.util.Date;
import java.util.List;

import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;

import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistrations;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ServiceRegistry;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class DeviceRegistrationDataService {

	private NotificationCommunicationDataLayer notificationCommunicationDataLayer;
	private DeviceRegistrationDataLayer datalayer;

	public DeviceRegistrationDataService() {
		datalayer = ServiceRegistry.getInstance().getPrimaryCommunity().getDataLayerFactory().getDeviceRegistrationDataLayer();
		notificationCommunicationDataLayer = ServiceRegistry.getInstance().getPrimaryCommunity().getDataLayerFactory().getNotificationCommunicationDataLayer();
	}
	
	public DeviceRegistrationDataService(DeviceRegistrationDataLayer datalayer, NotificationCommunicationDataLayer notificationSenderLayer) {
		this.datalayer = datalayer;
		this.notificationCommunicationDataLayer = notificationSenderLayer;
	}

	public DeviceRegistrations fetchRegistrations(String userId) {
		return datalayer.getRegistrations(userId);
	}

	public DeviceRegistration fetchRegistration(String userId, String deviceToken) {
		return datalayer.getRegistration(userId, deviceToken);
	}

	public DeviceRegistration saveRegistration(DeviceRegistration deviceRegistration) {
		if(deviceRegistration == null || NullChecker.isNullish(deviceRegistration.getDeviceToken())) {
			return null;
		}
		setNameOfCurrentUser(deviceRegistration);		
		synchronizeNamesOfExistingDeviceRegistrations(deviceRegistration);
		
		boolean exists = true;
		DeviceRegistration existingRegistration = null;
		
		existingRegistration = fetchRegistration(deviceRegistration.getUserId(), deviceRegistration.getDeviceToken());
		if (existingRegistration == null) {
			exists = false;
		} else {
			existingRegistration.setOptedIn(deviceRegistration.isOptedIn());
			return datalayer.saveRegistration(existingRegistration);
		}
		
		if(exists == false) {
			deviceRegistration.setRegisteredDate(new Date());
			return datalayer.saveRegistration(deviceRegistration);
		} 
		return existingRegistration;
	}
	
	public void updateOptStatus(DeviceRegistration deviceRegistration) {
		datalayer.updateOptStatus(deviceRegistration.getUserId(), deviceRegistration.isOptedIn());
	}

	private void synchronizeNamesOfExistingDeviceRegistrations(DeviceRegistration deviceRegistrationToMatch) {
		String userId = deviceRegistrationToMatch.getUserId();
		String firstName = deviceRegistrationToMatch.getFirstName();
		String lastName = deviceRegistrationToMatch.getLastName();

		DeviceRegistrations existingDeviceRegistrations = fetchRegistrations(userId);
		for (DeviceRegistration existingRegistration : existingDeviceRegistrations) {
			boolean modified = false;
			
			if (NullSafeStringComparer.areEqual(existingRegistration.getFirstName(),firstName) == false) {
				existingRegistration.setFirstName(firstName);
				modified = true;
			}
			if (NullSafeStringComparer.areEqual(existingRegistration.getLastName(), lastName) == false) {
				existingRegistration.setLastName(lastName);
				modified = true;
			}
			
			if (modified) {
				datalayer.saveRegistration(existingRegistration);
			}
		}
	}
	
	public void deleteRegistration(DeviceRegistration deviceRegistration) {
		datalayer.deleteRegistration(deviceRegistration);
	}
	
	public DeviceRegistrations fetchRegistrations(List<String> userIds) {
		return datalayer.fetchDeviceRegistrations(userIds);
	}
	
	/**
	 * Periodically call this method to clean up invalid tokens using Apple's
	 * Feedback Service
	 */
	public void cleanUpDeviceTokens() {
		List<String> invalidDeviceTokens = notificationCommunicationDataLayer.fetchInvalidDeviceTokens();
		for (String invalidDeviceToken : invalidDeviceTokens) {
			removeDeviceToken(invalidDeviceToken);
		}
	}

	private void removeDeviceToken(String deviceToken) {
		if (NullChecker.isNotNullish(deviceToken)) {
			datalayer.deleteRegistration(deviceToken);
		}
	}

	private void setNameOfCurrentUser(DeviceRegistration deviceRegistration) {
		deviceRegistration.setFirstName(getMhpUser().getFirstName());
		deviceRegistration.setLastName(getMhpUser().getLastName());
	}
	
	public DeviceRegistration deleteRegistration(String userId, String deviceToken) {
		DeviceRegistration deviceToUnregister = fetchRegistration(userId, deviceToken);
		
		deleteRegistration(deviceToUnregister);
		
		return deviceToUnregister;
	}
	
	private MhpUser getMhpUser(){
		return MhpUserFactory.createFromSecurityContext();
	}
}
