package com.agilex.healthcare.mobilehealthplatform.datalayer.medication.rxrefill;

import java.util.*;

import org.joda.time.Days;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.*;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.code.MedicationRefillCode;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.*;
import com.agilex.healthcare.utility.DateHelper;

public class MedicationRefillDataService {

	private Router router;
	private MedicationRefillRequestDataLayer medicationRefillRequestDataLayer;

	public MedicationRefillDataService() {
		this.router = new Router();
		medicationRefillRequestDataLayer = MhpObjectFactory.getInstance().getMedicationRefillRequestDataLayer();
	}

	public MedicationRefillDataService(Router router) {
		this.router = router;
	}

	public Medications fetchMedications(PatientIdentifier patientIdentifier) {
		RequestMessage request = MedicationRefillRequestBuilder.forFetchMedications().forPatientIdentifier(patientIdentifier).forDomain(Domain.medicationRefill).build();
		ResponseMessage response = router.execute(request);
		Medications medications = MedicationRefillResponseReader.fromResponse(response).getMedications();
		medications = emptyCollectionIfNull(medications);
		
		List<String> medicationIds = extractMedicationIds(medications);
		MedicationRefillRequests refillRequests = medicationRefillRequestDataLayer.fetchLatestRefillRequests(patientIdentifier, medicationIds);
		medications = reviewWithRequestRecords(medications, refillRequests);
		
		medications.sortAscending("refillAction");
		
		return medications;
	}

	public Medication refillMedication(Medication medication) {
			RequestMessage request = MedicationRefillRequestBuilder.forRefillMedication().forDomain(Domain.medicationRefill).forMedication(medication).build();
			ResponseMessage response = router.execute(request); 
			
			// If a refill can't be fulfilled and an expception is thrown, we don't record a refillRequest.
			
			MedicationRefillRequest refillRequest = new MedicationRefillRequest(medication.getPatientId(), medication.getUniqueId(), new Date());
			medicationRefillRequestDataLayer.saveRefillRequest(refillRequest);
			
			Medication refilledMedication = MedicationRefillResponseReader.fromResponse(response).getMedication();
			refilledMedication.setRefillAction(MedicationRefillCode.REFILL_ACTION_NONE);
			
			return refilledMedication;

	}

	Medications emptyCollectionIfNull(Medications medications) {
		if (medications == null) {
			return new Medications();
		}
		return medications;
	}
	
	void recordMedicationRefillRequest(Medication medication) {
		MedicationRefillRequest refillRequest = new MedicationRefillRequest(medication.getPatientId(), medication.getUniqueId(), new Date());
		medicationRefillRequestDataLayer.saveRefillRequest(refillRequest);
	}
	
	Medications reviewWithRequestRecords(Medications supposedlyRefillableMedications, MedicationRefillRequests refillRequests) {
		Medications filteredMedications = new Medications();
		for(Medication medication : supposedlyRefillableMedications) {
			if(MedicationRefillCode.REFILL_ACTION_NONE.equals(medication.getRefillAction())) { 
				filteredMedications.add(medication);
				continue;
			}
			Date comparisonDate = medication.getCalculatedNextFillDate();
			boolean found = false;
			for(MedicationRefillRequest refillRequest : refillRequests) {
				if(medication.getUniqueId().equals(refillRequest.getMedicationId())) {
					comparisonDate = nextRefillDate(refillRequest.getRequestDate(), medication.getDaysSupply());
					if(!DateHelper.greaterThanOrEqual(new Date(), comparisonDate)) {
						found = true;
					}
				}
			}
			
			if(found) 
				medication.setRefillAction(MedicationRefillCode.REFILL_ACTION_NONE);
			 
			
			filteredMedications.add(medication);
		}
		return filteredMedications;
	}
	
	List<String> extractMedicationIds(Medications medications) {
		List<String> medicationIds = new ArrayList<String>();
		
		for(Medication medication : medications) {
			medicationIds.add(medication.getUniqueId());
		}
		
		return medicationIds;
	}
	
	Date nextRefillDate(Date referenceDate, String daysSupply)  {
		MedicationRefillHelper helper = new MedicationRefillHelper();
		return helper.nextRefillDate(referenceDate, daysSupply);
	}
}
