package com.agilex.healthcare.mobilehealthplatform.datalayer.medication.rxrefill;

import java.util.*;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationRefillDataLayer;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.*;
import com.agilex.healthcare.mobilehealthplatform.utils.PropertyHelper;
import com.agilex.healthcare.utility.NullChecker;

public class FetchRefillableMedicationsRequestHandler implements RequestHandler {

	private static final String DFN = "dfn";
	private static final String ICN = "icn";
	public static final String SUPPORTED_SITES_PROPERTY = "facilities.rxrefill.pilotSites";
	private String supportedSites;
	
	public FetchRefillableMedicationsRequestHandler() {}
	
	public FetchRefillableMedicationsRequestHandler(PropertyHelper propertyHelper) {
		supportedSites = propertyHelper.getProperty(SUPPORTED_SITES_PROPERTY);
	}

	@Override
	public ResponseMessage handle(RequestMessage requestMessage) {
		MedicationRefillRequestReader requestReader = MedicationRefillRequestReader.fromRequest(requestMessage);

		PatientIdentifier currentPatientIdentifier = requestReader.getPatientIdentfier();
		
		// This code gets messy because we have to have the local sites associated with the patient
		// and the ICN (global VA identifier regardless of site). This doesn't fit most models, so we work
		// around it by using ICN-{siteCode} # ICN. This requires us to get all local identifiers (DFNs) associated
		// with the patient to build up the ICNs.
		
		// ICN PatientIdentifier for each will be equivalent to icn-{siteCode}#ICN#
		PatientIdentifiers icnIdentifiersForAllSites = constructICNIdentifiersForAllSites(currentPatientIdentifier);
		
		MedicationRefillDataLayer dataLayer = getDataLayer(requestMessage);
		
		PatientIdentifiers filteredIdentifiers = supportedIdentifiers(icnIdentifiersForAllSites);
		if(filteredIdentifiers.size() > 0) {
			Medications medications = invokeDataLayerForIdentifiers(dataLayer, filteredIdentifiers);
			return MedicationRefillResponseBuilder.fromMedications(medications).build();
		} else {
			return MedicationRefillResponseBuilder.fromMedications(new Medications()).build();
		}
	}

	private PatientIdentifiers constructICNIdentifiersForAllSites(PatientIdentifier currentPatientIdentifier) {
		PatientIdentifiers allIdentifiers = fecthAllPatientIdentifiers(currentPatientIdentifier);

		PatientIdentifiers allICNIdentifiers = null;
		PatientIdentifiers allDfnIdentifiers = null;
		PatientIdentifiers icnIdentifiers = new PatientIdentifiers();
		
		if(NullChecker.isNotNullish(allIdentifiers)){
			Map<String, PatientIdentifiers> identifiersGroupedByICNandDFN = groupIdentifiersByICNandDFN(allIdentifiers);
			allICNIdentifiers = identifiersGroupedByICNandDFN.get(ICN);
			allDfnIdentifiers = identifiersGroupedByICNandDFN.get(DFN);
		}

		if(NullChecker.isNotNullish(allICNIdentifiers) && NullChecker.isNotNullish(allDfnIdentifiers)){
			icnIdentifiers = constructICNIdentifierByAddingSiteCodeToAssigningAuthority(allICNIdentifiers, allDfnIdentifiers);
		}

		return icnIdentifiers;
	}

	private PatientIdentifiers constructICNIdentifierByAddingSiteCodeToAssigningAuthority(PatientIdentifiers allICNIdentifiers, PatientIdentifiers allDfnIdentifiers) {
		PatientIdentifiers icnIdentifiers = new PatientIdentifiers();
		
		for (PatientIdentifier dfnIdentifier : allDfnIdentifiers) {
			for (PatientIdentifier icnIdentifier : allICNIdentifiers) {
				icnIdentifiers.add(new PatientIdentifier(constructICNAssingingAuthorityWithSiteCode(dfnIdentifier.getAssigningAuthority()), icnIdentifier.getUniqueId()));
			}
		}
		
		return icnIdentifiers;
	}

	private String constructICNAssingingAuthorityWithSiteCode(String assigningAuthority) {
		//stripping of dfn from assigning authority
		return ICN + assigningAuthority.substring(3);
	}
	
	PatientIdentifiers supportedIdentifiers(PatientIdentifiers patientIdentifiers) {
		PatientIdentifiers filteredIdentifiers = new PatientIdentifiers();
		String[] sites = supportedSites.split(",");
		for(String site : sites) {
			String trimmedSite = site.trim();
			for(PatientIdentifier patientIdentifier : patientIdentifiers) {
				if(patientIdentifier.getAssigningAuthority().contains(trimmedSite)) {
					filteredIdentifiers.add(patientIdentifier);
				}
			}
		}
		return filteredIdentifiers;
	}

	private Map<String, PatientIdentifiers> groupIdentifiersByICNandDFN(PatientIdentifiers allIdentifiers) {
		Map<String, PatientIdentifiers> identifiersGroupedByType = new HashMap<String, PatientIdentifiers>();
		
		for (PatientIdentifier patientIdentifier : allIdentifiers) {
			if(patientIdentifier.getAssigningAuthority().toLowerCase().contains(ICN)){
				addIdentifierToTheGroup(identifiersGroupedByType, ICN, patientIdentifier);
			}else if(patientIdentifier.getAssigningAuthority().toLowerCase().contains(DFN)){
				addIdentifierToTheGroup(identifiersGroupedByType, DFN, patientIdentifier);
			}
		}
		
		return identifiersGroupedByType;
	}

	private void addIdentifierToTheGroup(Map<String, PatientIdentifiers> identifiersGroupedByType, String type, PatientIdentifier patientIdentifier) {
		PatientIdentifiers patientIdentifiers = identifiersGroupedByType.get(type);
		
		if(NullChecker.isNullish(patientIdentifiers)){
			patientIdentifiers = new PatientIdentifiers();
			identifiersGroupedByType.put(type, patientIdentifiers);
		}
		
		patientIdentifiers.add(patientIdentifier);
	}

	private PatientIdentifiers fecthAllPatientIdentifiers(PatientIdentifier currentPatientIdentifier) {
		PatientIdentifiers allIdentifiers = MhpObjectFactory.getInstance().getPatientCorrelationService()
				.getCorrespondIdentifiers(currentPatientIdentifier);
		
		return allIdentifiers;
	}

	private MedicationRefillDataLayer getDataLayer(RequestMessage requestMessage) {
		DomainServiceRegistry serviceRegistry = new DomainServiceRegistry();
		return serviceRegistry.getDataLayerByMessage(requestMessage);
	}

	Medications invokeDataLayerForIdentifiers(MedicationRefillDataLayer dataLayer, PatientIdentifiers patientIdentifiers) {
		Medications medications = new Medications();

		for (PatientIdentifier patientIdentifier : patientIdentifiers) {
			Medications fetched = dataLayer.fetchMedications(patientIdentifier);

			if(NullChecker.isNotNullish(fetched)){
				fixPatientIdentifier(fetched, patientIdentifier);
				medications.addAll(fetched);
			}
		}

		return medications;
	}

	private void fixPatientIdentifier(Medications fetched, PatientIdentifier patientIdentifier) {
		for (Medication medication : fetched) {
			medication.setPatientIdentifier(patientIdentifier);
		}
	}
}
