package com.agilex.healthcare.mobilehealthplatform.datalayer.lab.micro;

import java.util.HashMap;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.lab.*;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.mobilehealthplatform.domain.sorting.LabTestComparator;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.*;
import com.agilex.healthcare.mobilehealthplatform.utils.PatientIdentifierHelper;
import com.agilex.healthcare.utility.NullSafeStringComparer;

import edu.emory.mathcs.backport.java.util.Collections;

public class MicrobiologyLabDataService extends AbstractLabDataService {
	
	protected static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MicrobiologyLabDataService.class);

	private Router router;

	public MicrobiologyLabDataService() {
		this.router = new Router();
	}

	public MicrobiologyLabDataService(Router router) {
		this.router = router;
	}

	public LabResults getLabResultsByTestAndSpecimen(PatientIdentifier patientIdentifier, String testId, String specimenId, ScopeFilter scopeFilter) {
		// TODO: the retrieve / filter should likely be a specific message for
		// labs
//		logger.info(String.format("retrieve lab results [patient=%s][test=%s][specimen=%s]", patientIdentifier, testId, specimenId));
		LabResults results = this.getLabResultsWithTextResults(patientIdentifier, getDefaultDateFilter(), scopeFilter);
		logger.debug(String.format("retrieved %s lab results", results.size()));
		LabResults filteredList = LabResultsFilter.filter(results, testId, specimenId);
		logger.debug(String.format("filtered list by testid, filtered count=%s ", filteredList.size()));
		calculateResultAbnormalityFlag(filteredList);
//		logger.info(String.format("retrieved %s lab results [patient=%s][test=%s][specimen=%s]", filteredList.size(), patientIdentifier, testId, specimenId));

		PatientIdentifierHelper.updatePatientIdentifier(filteredList, patientIdentifier);

		return filteredList;
	}

	public LabTests getLabTestsWithMostRecentResult(PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
//		logger.info(String.format("retrieve lab tests [patient=%s]", patientIdentifier));
		LabResults results = this.getLabResultsWithTextResults(patientIdentifier, getDefaultDateFilter(), scopeFilter);
		LabTests labTests = null;

		if (results != null) {
			calculateResultAbnormalityFlag(results);

			// map to store the most recent lab result for a given lab test
			LabTestFilterByMostRecentResult filter = new LabTestFilterByMostRecentResult();
			HashMap<String, LabResult> labResultsIndexedByTestId = filter.getMostRecentResultsByTest(results);
			labTests = getLabTestsFromResults(labResultsIndexedByTestId);
		} else {
			labTests = new LabTests();
		}

//		logger.info(String.format("retrieved %s lab tests [patient=%s]", labTests.size(), patientIdentifier));
		return labTests;
	}
	
	LabTests getLabTests(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		LabResults labResults = this.getLabResultsWithTextResults(patientIdentifier, dateFilter, scopeFilter);
		
		LabTests labTests = new LabTests();
		for (LabResult result : labResults) {
			LabTest labTest = new LabTest();

			labTest.getLabResults().add(result);
			labTest.setUniqueId(result.getTestId());
			labTest.setTestname(result.getTestname());
			labTest.setSpecimenId(result.getSpecimenId());
			labTest.setSpecimenName(result.getSpecimenName());

			labTests.add(labTest);
		}
		
		return labTests;
	}

	public LabTests getResultsGroupedByTest(PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
//		logger.info(String.format("retrieve lab tests [patient=%s]", patientIdentifier));
		LabResults results = this.getLabResultsWithoutTextResults(patientIdentifier, getDefaultDateFilter(), scopeFilter);
		determineDisplayName(results);

		LabTests labTests = null;

		if (results != null) {
			calculateResultAbnormalityFlag(results);
			LabResultOrganizer grouping = new LabResultOrganizer();
			labTests = grouping.groupByTest(results);
		} else {
			labTests = new LabTests();
		}

//		logger.info(String.format("retrieved %s lab tests [patient=%s]", labTests.size(), patientIdentifier));
		Collections.sort(labTests, new LabTestComparator());
		return labTests;
	}

	private LabTests getLabTestsFromResults(HashMap<String, LabResult> labTestHistory) {

		LabTests labTests = new LabTests();

		for (LabResult result : labTestHistory.values()) {
			LabTest labTest = new LabTest();
			LabResults labResults = new LabResults();
			labResults.add(result);

			labTest.setLabResults(labResults);
			labTest.setUniqueId(result.getTestId());
			labTest.setTestname(result.getTestname());
			labTest.setSpecimenId(result.getSpecimenId());
			labTest.setSpecimenName(result.getSpecimenName());

			labTests.add(labTest);
		}
		
		return labTests;
	}

	public LabResults getLabOrderResults(PatientIdentifier patientIdentifier, String orderId, ScopeFilter scopeFilter) {
//		logger.info(String.format("retrieve lab results [patient=%s][orderid=%s]", patientIdentifier, orderId));
		LabResults results = this.getLabResultsWithTextResults(patientIdentifier, getDefaultDateFilter(), scopeFilter);
		LabResults filteredResults = LabResultsFilterByOrderNumber.filter(results, orderId);
//		logger.info(String.format("retrieved %s lab results [patient=%s][orderid=%s]", filteredResults.size(), patientIdentifier, orderId));
		return filteredResults;
	}

	/**
	 * This gets gets the LabTests for MostRecentResult and then calls
	 * extractAndSortLabGroupsByLabTests
	 * 
	 * @param scopeFilter
	 * 
	 * @see #extractAndSortLabGroupsByLabTests(LabTests)
	 * 
	 * @param patientId
	 * @return
	 */
	public LabTestGroups getLabTestsInGroups(PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
//		logger.info(String.format("begin getLabTestsInGroups [patient=%s]", patientIdentifier));
		LabTests labTests = getLabTests(patientIdentifier, DateFilterFactory.createEmptyFilter(), scopeFilter);
//		logger.info(String.format("retrieved lab tests [patient=%s][test.count=%s]", patientIdentifier, labTests.size()));
		LabTestGroups labTestGroups = groupTests(labTests);
//		logger.info(String.format("extracted / sorted lab tests [patient=%s][group.count=%s]", patientIdentifier, labTestGroups.size()));

		return labTestGroups;
	}

	public LabResult getMicrobiologyLabResult(PatientIdentifier patientIdentifier, String resultId, ScopeFilter scopeFilter) {
		LabResults allResults = getLabResultsWithTextResults(patientIdentifier, DateFilterFactory.createEmptyFilter(), scopeFilter);
		LabResult matchedLabResult = null;
		for (LabResult possibleMatch : allResults) {
			if (NullSafeStringComparer.areEqual(possibleMatch.getUniqueId(), resultId)) {
				matchedLabResult = possibleMatch;
				break;
			}
		}
		return matchedLabResult;
	}
	
	private LabTestGroups groupTests(LabTests labTests) {
		MicrobiologyLabGrouper grouper = new MicrobiologyLabGrouper();
		return grouper.groupAndSort(labTests);
	}
	
	public LabResults getLabResultsWithTextResults(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
//		logger.debug(String.format("retrieve lab results [patient=%s]", patientIdentifier));
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.microLabResult).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).forDateFilter(dateFilter).build();
		ResponseMessage responseMessage = router.execute(request);
		LabResults labResults = PatientDataResponseReader.<LabResults, LabResult> fromResponse(responseMessage).getDataListNoNull(LabResults.class);
		determineDisplayName(labResults);
		labResults.sortDescending("resultedDate");
//		logger.debug(String.format("retrieved %s lab results [patient=%s]", labResults.size(), patientIdentifier));
		return labResults;
	}

	public LabResults getLabResultsWithoutTextResults(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		return getLabResultsWithTextResults(patientIdentifier, dateFilter, scopeFilter);
	}

	public LabResults getLabResults(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		return getLabResultsWithTextResults(patientIdentifier, dateFilter, scopeFilter);
	}

}
