package com.agilex.healthcare.mobilehealthplatform.datalayer.lab;

import com.agilex.healthcare.mobilehealthplatform.domain.LabTest;
import com.agilex.healthcare.mobilehealthplatform.domain.LabTestGroup;
import com.agilex.healthcare.mobilehealthplatform.domain.LabTestGroups;
import com.agilex.healthcare.mobilehealthplatform.domain.LabTests;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.utility.NullChecker;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class ChemistryLabGrouper {

	private HashMap<LabTestGroup, List<String>> groupDefinitions;
	Map<String, LabTest> groupedTests = new ConcurrentHashMap<String, LabTest>();

	public ChemistryLabGrouper() {
		MhpObjectFactory mhpFactory = new MhpObjectFactory();
		this.groupDefinitions = mhpFactory.chemistryLabGroupDefinitions();
	}

	public LabTestGroups group(LabTests labTests) {
		if (NullChecker.isNullish(labTests))
			return new LabTestGroups();

		Map<String, LabTests> mappedTests = mapByGroupingId(labTests);
		return groupMappedTests(mappedTests);
	}

	private LabTestGroups groupMappedTests(Map<String, LabTests> mappedTests) {
		LabTestGroups groups = new LabTestGroups();

		// pull out the most recent of each type of test (this single lab will
		// be displayed to the user)
		Map<String, LabTest> mostRecentMappedTests = mapMostRecent(mappedTests);
		for (LabTestGroup groupDefinition : groupDefinitions.keySet()) {
			LabTestGroup labTestGroup = fillGroupWithMatchingTests(groupDefinition, mostRecentMappedTests);
			if (labTestGroup.getLabTests() != null && labTestGroup.getLabTests().size() > 0) {
				groups.add(labTestGroup);
			}
		}

		LabTestGroup ungroupedTests = findAndOrganizeUngroupedTests(mostRecentMappedTests);
		groups.add(ungroupedTests);

		return groups;
	}

	private LabTestGroup fillGroupWithMatchingTests(LabTestGroup labTestGroup, Map<String, LabTest> mostRecentMappedTests) {
		List<String> testIdentifiers = groupDefinitions.get(labTestGroup);
		LabTestGroup group = new LabTestGroup(labTestGroup.getName()); // don't mutate the definition's key; create a copy

		// iterate through the defined identifiers for the lab test group and
		// add any matching lab Tests
		for (String identifier : testIdentifiers) {
			String[] groupDefinitionIds = identifier.split(",");
			for (String groupDefinitionId : groupDefinitionIds) {
				LabTest labTest = mostRecentMappedTests.get(groupDefinitionId);
				if (labTest != null) {
					groupedTests.put(groupDefinitionId, labTest);
					group.addLabTest(labTest);
				}
			}
			// ignore lab tests that don't get mapped for now - we will scan for those at the end.
		}

		return group;
	}

	private Map<String, LabTests> mapByGroupingId(LabTests labTests) {
		Map<String, LabTests> mappedTests = new ConcurrentHashMap<String, LabTests>();

		for (LabTest labTest : labTests) {
			String groupId = labTest.getGroupingId();

			LabTests tests = mappedTests.get(groupId);
			if (labTests == null) {
				tests.add(labTest);
			} else {
				mappedTests.put(groupId, new LabTests(labTest));
			}
		}

		return mappedTests;
	}

	Map<String, LabTest> mapMostRecent(Map<String, LabTests> mappedTests) {
		Map<String, LabTest> mapOfMostRecent = new HashMap<String, LabTest>();

		for (String key : mappedTests.keySet()) {
			LabTests labTests = mappedTests.get(key);
			Collections.sort(labTests, new LabTestNameComparator());
			mapOfMostRecent.put(key, labTests.get(0));
		}

		return mapOfMostRecent;
	}

	LabTestGroup findAndOrganizeUngroupedTests(Map<String, LabTest> mappedTests) {
		LabTestGroup ungroupedTests = new LabTestGroup("Ungrouped");
		for (String key : mappedTests.keySet()) {
			LabTest alreadyMapped = groupedTests.get(key);
			if (alreadyMapped == null) {
				ungroupedTests.addLabTest(mappedTests.get(key));
			}
		}

        Collections.sort(ungroupedTests.getLabTests(), new LabTestNameComparator());
		return ungroupedTests;
	}
}
