package com.agilex.healthcare.mobilehealthplatform.datalayer.lab;

import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.facility.FacilityCache;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.*;
import com.agilex.healthcare.mobilehealthplatform.utils.PatientIdentifierHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.vamf.cdw.chemlabs.CdwChemistryLabDataLayer;

public class ChemistryLabDataService extends AbstractLabDataService {

	protected static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ChemistryLabDataService.class);
	private Router router;
	private CdwChemistryLabDataLayer layer;

	public ChemistryLabDataService() {
		this.router = new Router();
		layer = MhpObjectFactory.getInstance().getCdwChemistryLabDataLayer();
	}

	public ChemistryLabDataService(Router router) {
		this.router = router;
		layer = MhpObjectFactory.getInstance().getCdwChemistryLabDataLayer();
	}

	public LabResults getLabResults(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		/*RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.chemLabResult).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).forDateFilter(dateFilter).build();
		ResponseMessage responseMessage = router.execute(request);
		LabResults labResults = PatientDataResponseReader.<LabResults, LabResult> fromResponse(responseMessage).getDataListNoNull(LabResults.class);*/
        PatientIdentifiers icnPatientIdentifiers = PatientIdentifierHelper.translatePatientIdentifier(patientIdentifier, "ICN");
        if(NullChecker.isNullish(icnPatientIdentifiers))
        	return new LabResults();
        LabResults labResults = layer.getLabResults(icnPatientIdentifiers.get(0), dateFilter);
        PatientIdentifierHelper.updatePatientIdentifier(labResults, patientIdentifier);
		convertFacility(labResults);

        labResultsGraphable(labResults);
		calculateResultAbnormalityFlag(labResults);
		labResults.sortDescending("resultedDate");

		return labResults;
	}

	public LabResults getLabResults(PatientIdentifier patientIdentifier, String groupingId, DateFilter dateFilter, ScopeFilter scopeFilter) {
		LabResults results = getLabResults(patientIdentifier, dateFilter, scopeFilter);
		return filter(results, groupingId);
	}

	private LabResults filter(LabResults labResults, String groupingId) {
		LabResultsFilter filter = new LabResultsFilter();
		return filter.filter(labResults, groupingId);
	}
	
	public LabTests getLabTestsWithMostRecentResult(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		LabResults labResults = getLabResults(patientIdentifier, dateFilter, scopeFilter);

		LabTests labTests;
		if (NullChecker.isNotNullish(labResults)) {
			// map to store the most recent lab result for a given lab test
			LabTestFilterByMostRecentResult filter = new LabTestFilterByMostRecentResult();
			HashMap<String, LabResult> labResultsIndexedByTestId = filter.getMostRecentResultsByTest(labResults);
			labTests = getLabTestsFromResults(labResultsIndexedByTestId);
		} else {
			labTests = new LabTests();
		}

		return labTests;
	}

	/**
	 * This gets gets the LabTests for MostRecentResult and then calls
	 * extractAndSortLabGroupsByLabTests
	 * 
	 * @see #extractAndSortLabGroupsByLabTests(LabTests)
	 * 
	 * @param patientIdentifier
	 * @return
	 */
	public LabTestGroups getLabTestsInGroups(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		LabTests labTests = getLabTestsWithMostRecentResult(patientIdentifier, dateFilter, scopeFilter);
		LabTestGroups labTestGroups = extractAndSortLabGroupsByLabTests(labTests);

		return labTestGroups;
	}

	private LabTests getLabTestsFromResults(HashMap<String, LabResult> labTestHistory) {
		LabTests labTests = new LabTests();

		for (LabResult result : labTestHistory.values()) {
			LabTest labTest = new LabTest();
			labTest.setMostRecentLabResult(result);
			labTest.setUniqueId(result.getTestId());
			labTest.setDataIdentifier(result.getDataIdentifier().getSystemId(), result.getTestId());
			labTest.setPatientIdentifier(result.getPatientIdentifier());
			labTest.setTestname(result.getTestname());
			labTest.setSpecimenId(result.getSpecimenId());
			labTest.setSpecimenName(result.getSpecimenName());
			labTest.setTestDescription(result.getDisplayDescription());
			labTest.setLoinc(result.getLoinc());

			labTests.add(labTest);
		}
		return labTests;
	}

    private void convertFacility(LabResult result) {
        String facilityName = FacilityCache.getInstance().getFacilityName(result.getSourceSystem());
        result.setSourceSystem(facilityName);
    }

    private void convertFacility(LabResults results) {
        for (LabResult result : results) {
            convertFacility(result);
        }
    }

    // Check if the labResult value field contains only Text
    // if so sets the graphable to false else to true
    void labResultsGraphable(LabResults labResults) {
        for (LabResult labResult : labResults) {
            try {
                Double.parseDouble(labResult.getValue());

                labResult.setGraphable(true);
            } catch(NumberFormatException formatException) {
                labResult.setGraphable(false);
            } catch(NullPointerException nullPointerException) {
                labResult.setGraphable(false);
            }
        }
    }
}
