package com.agilex.healthcare.mobilehealthplatform.datalayer.healthadvocaterequest;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import org.apache.log4j.Logger;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataEditRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataResponseReader;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.DateHelper;

import edu.emory.mathcs.backport.java.util.Collections;

public class HealthAdvocateRequestDataService {

	private static final Logger logger = Logger.getLogger(HealthAdvocateRequestDataService.class);
	private Router router = new Router();

	public HealthAdvocateRequestDataService() {
	}

	public HealthAdvocateRequests fetchHealthAdvocateRequests(PatientIdentifier patientIdentifier) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.healthAdvocateRequest).forPatientIdentifier(patientIdentifier)
				.forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);
		HealthAdvocateRequests healthAdvocateRequests = PatientDataResponseReader.<HealthAdvocateRequests, HealthAdvocateRequest> fromResponse(responseMessage).getDataListNoNull(
				HealthAdvocateRequests.class);
		HealthAdvocateRequests filteredHealthAdvocateRequests = new HealthAdvocateRequests();

		filteredHealthAdvocateRequests = filterHealthAdvocateRequests(healthAdvocateRequests);
		sortHealthAdvocateRequests(filteredHealthAdvocateRequests);

		return filteredHealthAdvocateRequests;

	}

	private HealthAdvocateRequests filterHealthAdvocateRequests(HealthAdvocateRequests healthAdvocateRequests) {
		HealthAdvocateRequests filteredHealthAdvocateRequests = new HealthAdvocateRequests();

		Date today = new Date();

		for (HealthAdvocateRequest healthAdvocateRequest : healthAdvocateRequests) {
			if (healthAdvocateRequest.getStatus().equalsIgnoreCase(HealthAdvocateStatusCode.SUBMITTED) || healthAdvocateRequest.getStatus().equalsIgnoreCase(HealthAdvocateStatusCode.ACCEPTED)) {
				filteredHealthAdvocateRequests.add(healthAdvocateRequest);
				continue;
			} else if (DateHelper.calculateDeltaInDays(healthAdvocateRequest.getLastActionDate(), today) < 180) {
				filteredHealthAdvocateRequests.add(healthAdvocateRequest);
			}
		}

		return filteredHealthAdvocateRequests;
	}

	private void sortHealthAdvocateRequests(HealthAdvocateRequests healthAdvocateRequests) {
		Collections.sort(healthAdvocateRequests, new HealthAdvocateRequestSortByStatusAndLastActionDateComparator());
	}

	public HealthAdvocateRequest fetchHealthAdvocateRequest(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveSingleById().forDomain(Domain.healthAdvocateRequest).forPatientIdentifier(patientIdentifier)
				.forDataIdentifier(dataIdentifier).build();
		ResponseMessage responseMessage = router.execute(request);
		return PatientDataResponseReader.<HealthAdvocateRequests, HealthAdvocateRequest> fromResponse(responseMessage).getDataItem();
	}

	public HealthAdvocateRequest createNewHealthAdvocateRequest(PatientIdentifier patientIdentifier, HealthAdvocateRequest healthAdvocateRequest) {
		logger.debug("creating a new health advocate request");

		RequestMessage request = PatientDataEditRequestBuilder.forCreate().forDomain(Domain.healthAdvocateRequest).forData(healthAdvocateRequest).forPatientIdentifier(patientIdentifier)
				.forScope(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);
		HealthAdvocateRequest createdHealthAdvocateRequest = PatientDataResponseReader.<HealthAdvocateRequests, HealthAdvocateRequest> fromResponse(responseMessage).getDataItem();

		return createdHealthAdvocateRequest;
	}

	public HealthAdvocateRequest updateHealthAdvocateRequest(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier, HealthAdvocateRequest healthAdvocateRequest) {
		logger.debug("updating a new health advocate request");

		healthAdvocateRequest.setLastActionDate(DateHelper.getToday());
		
		RequestMessage request = PatientDataEditRequestBuilder.forUpdate().forDomain(Domain.healthAdvocateRequest).forData(healthAdvocateRequest).forDataIdentifier(dataIdentifier)
				.forPatientIdentifier(patientIdentifier).forScope(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);
		HealthAdvocateRequest updatedHealthAdvocateRequest = PatientDataResponseReader.<HealthAdvocateRequests, HealthAdvocateRequest> fromResponse(responseMessage).getDataItem();

		updatedHealthAdvocateRequest.setPatientIdentifier(patientIdentifier);
		updatedHealthAdvocateRequest.setDataIdentifier(dataIdentifier);
		return updatedHealthAdvocateRequest;
	}

    public HealthAdvocateRequestForm saveHealthAdvocateRequestForm(PatientIdentifier patientIdentifier, HealthAdvocateRequestForm healthAdvocateRequestForm) {
        logger.debug("saving health advocate request form");

        RequestMessage request = PatientDataEditRequestBuilder.forUpdate().forDomain(Domain.healthAdvocateRequestForm).forData(healthAdvocateRequestForm).forPatientIdentifier(patientIdentifier).forScope(ScopeFilter.getInstanceForLongitudinalScope()).build();
        ResponseMessage responseMessage = router.execute(request);
        return PatientDataResponseReader.<HealthAdvocateRequestForms, HealthAdvocateRequestForm> fromResponse(responseMessage).getDataItem();
    }

    public HealthAdvocateRequestEligibility fetchHealthAdvocateRequestEligibility (PatientIdentifier patientIdentifier) {
	
		logger.debug("checking if eligible for a new health advocate request");

		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.healthAdvocateRequest).forPatientIdentifier(patientIdentifier)
				.forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);
		HealthAdvocateRequests healthAdvocateRequests = PatientDataResponseReader.<HealthAdvocateRequests, HealthAdvocateRequest> fromResponse(responseMessage).getDataListNoNull(
				HealthAdvocateRequests.class);
		HealthAdvocateRequests filteredHealthAdvocateRequests = new HealthAdvocateRequests();

		filteredHealthAdvocateRequests = filterHealthAdvocateRequests(healthAdvocateRequests);
		
		HealthAdvocateRequestEligibility healthAdvocateRequestEligibility = calculateEligibility(patientIdentifier, filteredHealthAdvocateRequests);

		return healthAdvocateRequestEligibility;
	}

	private HealthAdvocateRequestEligibility calculateEligibility(PatientIdentifier patientIdentifier, HealthAdvocateRequests filteredHealthAdvocateRequests) {
		HealthAdvocateRequestEligibility eligibility = new HealthAdvocateRequestEligibility(patientIdentifier);
		eligibility.setEligible(true);
		eligibility.setMessage("Eligible.");
		
		for (HealthAdvocateRequest har : filteredHealthAdvocateRequests) {
			if ((har.getStatus().equalsIgnoreCase(HealthAdvocateStatusCode.SUBMITTED) || (har.getStatus().equalsIgnoreCase(HealthAdvocateStatusCode.ACCEPTED)))) {
				eligibility.setEligible(false);
				eligibility.setMessage("You cannot request another Health Advocate. There is already at least one other request with a status of " + har.getStatus() + ".");
				break;
			}
		}
		
		return eligibility;
	}



}
