package com.agilex.healthcare.mobilehealthplatform.datalayer.feedback;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.UserFeedback;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.restservice.exception.InvalidUserFeedbackException;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;

public class UserFeedbackDataService {
	
	private static final String APPOINTMENT_PROCESSED_AS_PER_USER_FIELD = "appointmentProcessedAsPerUserNeed";
	private static final String RECOMMENDATION_RATING_FIELD = "recommendationRating";
	private static final String ERROR = "error";

	public UserFeedback saveUserFeedback(UserFeedback userFeedback) {
		validateUserFeedback(userFeedback);
		userFeedback.setDateCreated(new Date());
		return MhpObjectFactory.getInstance().getUserFeedbackDataLayer().saveUserFeedback(userFeedback);
	}

	private void validateUserFeedback(UserFeedback userFeedback) {
		ValidationResult<UserFeedback> validationResult = new ValidationResult<UserFeedback>();
		validationResult.setRequestObject(userFeedback);

		if(userFeedback == null){
			validationResult.addIfNotNull(createValidationError(ERROR, "UserFeedback object is empty"));
		}else{
			validateUserSatisfactionEntry(userFeedback, validationResult);
			validateRecommendationRating(userFeedback, validationResult);
		}
		
		if (!validationResult.isValid()) {
			throw new InvalidUserFeedbackException(validationResult.getErrors());
		}
	}
	
	private void validateRecommendationRating(UserFeedback userFeedback, ValidationResult<UserFeedback> validationResult) {
		if(userFeedback.getRecommendationRating() == null){
			validationResult.addIfNotNull(createValidationError(RECOMMENDATION_RATING_FIELD, "Recommendation rating field is not set"));
		}
	}

	private void validateUserSatisfactionEntry(UserFeedback userFeedback, ValidationResult<UserFeedback> validationResult) {
		if(userFeedback.getAppointmentProcessedAsPerUserNeed() == null){
			validationResult.addIfNotNull(createValidationError(APPOINTMENT_PROCESSED_AS_PER_USER_FIELD, "Appointment Processed as per patient requirements is not set"));
		}
	}

	private ValidationError createValidationError(String field, String message){
		ValidationError error = new ValidationError();
		error.setErrorMessage(message);
		error.setFieldName(field);
		return error;
	}

}
