package com.agilex.healthcare.mobilehealthplatform.datalayer.document;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataEditRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataResponseReader;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Document;
import com.agilex.healthcare.mobilehealthplatform.domain.DocumentBody;
import com.agilex.healthcare.mobilehealthplatform.domain.Documents;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.RelatedDataItem;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.NullChecker;

public class DocumentService {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(DocumentService.class);

	private Router router;

	public DocumentService() {
		router = new Router();
	}

	public DocumentService(Router router) {
		router = new Router();
	}

	public Documents fetchDocumentsForPatient(PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
		return fetchDocumentsForPatient(patientIdentifier, null, scopeFilter);
	}

	public Documents fetchDocumentsForPatient(PatientIdentifier patientIdentifier, DocumentFilter filter, ScopeFilter scopeFilter) {
		//logger.debug(String.format("begin fetch documents for patient with filter [p=%s][filter=%s]", patientIdentifier, filter));

		logger.debug("begin fetch documents for patient");

		RequestMessage request = FetchDocumentListRequestBuilder.forRetrieveList().forDocumentFilter(filter).forDomain(Domain.document).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).build();
		ResponseMessage responseMessage = router.execute(request);
		Documents documents = PatientDataResponseReader.<Documents, Document> fromResponse(responseMessage).getDataListNoNull(Documents.class);

		logger.debug("complete with raw document fetch");

		// note: this was added in case the data layer doesn't apply the filter.
		// This could be removed and put into the data layer or handler
//		Documents filteredDocuments = DocumentFilterer.filter(documents, filter);
//		logger.debug(String.format("applied document filter [rc=%s]", filteredDocuments.size()));
		return documents;
	}

	public Documents fetchDocumentsForPatientAssociatedWithPatientData(PatientIdentifier patient, String relatedDataDomain, DataIdentifier relatedDataIdentifier, ScopeFilter scopeFilter) {
		logger.debug("begin fetch documents for associated patient");

		DocumentFilter filter = new DocumentFilter();
		filter.setParentItem(new RelatedDataItem(relatedDataDomain, relatedDataIdentifier));

		Documents documents = fetchDocumentsForPatient(patient, filter, scopeFilter);

		logger.debug("complete fetch documents for patient");
		return documents;
	}

	public Document fetchSingleDocument(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier) {
		logger.debug("begin fetch document");

		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveSingleById().forDomain(Domain.document).forPatientIdentifier(patientIdentifier).forDataIdentifier(dataIdentifier).build();
		ResponseMessage responseMessage = router.execute(request);
		Document document = PatientDataResponseReader.<Documents, Document> fromResponse(responseMessage).getDataItem();

		logger.debug("complete with fetch document");
		return document;
	}

	public DocumentBody fetchSingleDocumentBody(PatientIdentifier patient, DataIdentifier dataIdentifier) {
		Document document = fetchSingleDocument(patient, dataIdentifier);
		DocumentBody documentBody = null;
		if (document != null)
			documentBody = document.getBody();
		return documentBody;
	}

	public Document createDocument(Document document, ScopeFilter scopeFilter) {
		RequestMessage request = PatientDataEditRequestBuilder.forCreate().forDomain(Domain.document).forData(document).forScope(scopeFilter).build();
		ResponseMessage responseMessage = router.execute(request);
		Document savedDocument = PatientDataResponseReader.<Documents, Document> fromResponse(responseMessage).getDataItem();
		return savedDocument;
	}

	public Document updateDocument(Document document) {
		RequestMessage request = PatientDataEditRequestBuilder.forUpdate().forDomain(Domain.document).forData(document).build();
		ResponseMessage responseMessage = router.execute(request);
		Document savedDocument = PatientDataResponseReader.<Documents, Document> fromResponse(responseMessage).getDataItem();
		return savedDocument;
	}

	public void deleteDocument(Document document) {
		RequestMessage request = PatientDataEditRequestBuilder.forDelete().forDomain(Domain.document).forData(document).build();
		ResponseMessage responseMessage = router.execute(request);
		PatientDataResponseReader.<Documents, Document> fromResponse(responseMessage).getDataItem();
	}

	public Document createDocumentAssociatedWithPatientData(Document document, String relatedDataDomain, DataIdentifier relatedDataIdentifier, ScopeFilter scopeFilter) {
		logger.debug("save document associated with patient data");
		RelatedDataItem parent = new RelatedDataItem();
		parent.setDomain(relatedDataDomain);
		parent.setIdentifier(relatedDataIdentifier);
		document.setParentData(parent);

		if (NullChecker.isNullish(document.getDocumentTitle())) {
			document.setDocumentTitle("Document associated with " + relatedDataDomain + ";" + relatedDataIdentifier);
		}

		return createDocument(document, scopeFilter);
	}

	public Document saveDocumentContent(PatientIdentifier patientIdentifier, DataIdentifier documentIdentifier, byte[] documentContent, String contentType) {
		logger.debug("fetch document");
		Document document = fetchSingleDocument(patientIdentifier, documentIdentifier);
		logger.debug("fetched document");
		logger.debug("assign content to document");
		document.getBody().setContents(documentContent);
		document.getBody().setMimeType(contentType);
		document = updateDocument(document);
		logger.debug("saved document");
		return document;
	}

}
