package com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata;

import java.util.Collection;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestHandler;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DataSystem;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DomainServiceRegistry;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.utility.NullChecker;

public class SavePatientDataRequestHandler implements RequestHandler {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(SavePatientDataRequestHandler.class);

	@Override
	public ResponseMessage handle(RequestMessage requestMessage) {
		logger.debug("received request message");

		PatientDataEditRequestReader<PatientData> messageReader = PatientDataEditRequestReader.fromRequest(requestMessage);
//		logger.debug(String.format("request=>[data=%s][patient=%s][dataid=%s][scope=%s]", messageReader.getData(), messageReader.getPatientIdentifier(), messageReader.getDataIdentifier(), messageReader.getScopeFilter()));

		// TODO: consider how to differentiate between save new / update /
		// delete
		String childMessageType;
		if (requestMessage.getType().contentEquals("CreatePatientData")) {
			childMessageType = messageReader.getDomain() + ".create";
		} else if (requestMessage.getType().contentEquals("UpdatePatientData")) {
			childMessageType = messageReader.getDomain() + ".update";
		} else if (requestMessage.getType().contentEquals("DeletePatientData")) {
			childMessageType = messageReader.getDomain() + ".delete";
		} else {
			childMessageType = messageReader.getDomain() + requestMessage.getType();
		}
		requestMessage.setType(childMessageType);

		DataSystem system = determineSystem(messageReader);
		PatientIdentifiers translatedPatientIdentifiers = getTranslatedPatientIdentifiers(messageReader, system);

		if (NullChecker.isNullish(translatedPatientIdentifiers)) {
			throw new RuntimeException("Unable to deteremine destination patient identifier, therefore cannot save patient data");
		}
		
		PatientIdentifier translatedPatientIdentifier = translatedPatientIdentifiers.get(0);

		RequestMessage childRequestMessage = PatientDataEditRequestBuilder.cloneFromParent(requestMessage).forType(childMessageType).forTargetDataSystem(system).forData(messageReader.getData()).forPatientIdentifier(translatedPatientIdentifier)
				.build();

		logger.debug("passing request for patient data to router to be routed to domain specific handler");
		Router router = new Router();
		ResponseMessage response = router.execute(childRequestMessage);

		fixDataIdentifier(response, system);
		fixPatientIdentifier(response, translatedPatientIdentifier);

		logger.debug("complete with execute request message");
		return response;
	}

	private void fixDataIdentifier(ResponseMessage response, DataSystem system) {
		PatientDataResponseReader<?, ?> responseReader = PatientDataResponseReader.fromResponseWithoutType(response);
		PatientData dataItem = responseReader.getDataItem();
		if (dataItem != null && NullChecker.isNullish(dataItem.getDataIdentifier().getSystemId())) {
			logger.debug("saved item");
			dataItem.getDataIdentifier().setSystemId(system.getSystemIdentifier());
			logger.debug("change saved item");
		}
	}

	private void fixPatientIdentifier(ResponseMessage response, PatientIdentifier translatedPatientIdentifier) {
		PatientDataResponseReader<?, ?> responseReader = PatientDataResponseReader.fromResponseWithoutType(response);
		PatientData dataItem = responseReader.getDataItem();

		if (dataItem != null) {
			PatientIdentifier savedDataPatientIdentifier = dataItem.getPatientIdentifier();
			PatientIdentifier originalPatientIdentifier = PatientDataEditRequestReader.fromRequest(responseReader.getRequestReader().request).getData().getPatientIdentifier();

			logger.debug("checking patient identifier => " + savedDataPatientIdentifier);
			if (NullChecker.isNullish(savedDataPatientIdentifier.getAssigningAuthority())) {
				responseReader.getDataItem().setPatientIdentifier(originalPatientIdentifier);
				logger.debug("fix patient identifier => " + responseReader.getDataItem().getPatientIdentifier());
			}
		}
	}

	private PatientIdentifiers getTranslatedPatientIdentifiers(PatientDataEditRequestReader<PatientData> messageReader, DataSystem dataSystem) {

		PatientIdentifier patientIdentifier = messageReader.getData().getPatientIdentifier();

		PatientIdentifiers translatedPatientIdentifiers = null;
		if (dataSystem.getPatientAssigningAuthority().contentEquals(patientIdentifier.getAssigningAuthority())) {
			translatedPatientIdentifiers = new PatientIdentifiers();
			translatedPatientIdentifiers.add(patientIdentifier);
		} else {
			translatedPatientIdentifiers = translatePatientIdentifier(patientIdentifier, dataSystem.getPatientAssigningAuthority());
		}
		return translatedPatientIdentifiers;
	}

	// TODO: review the logic to determine which data system to save this to
	private DataSystem determineSystem(PatientDataEditRequestReader<PatientData> messageReader) {
		DomainServiceRegistry registry = new DomainServiceRegistry();

		DataSystem dataSystem = null;
		DataIdentifier dataIdentifier = null;
		if (messageReader.getData() != null && messageReader.getData().getDataIdentifier() != null) {
			dataIdentifier = messageReader.getData().getDataIdentifier();
		}

		if (dataIdentifier != null && NullChecker.isNotNullish(dataIdentifier.getSystemId())) {
			dataSystem = registry.getDataSystemByName(messageReader.getData().getDataIdentifier().getSystemId());
		}

		if (dataSystem == null) {
			Collection<DataSystem> dataSystems = registry.getDataSystems(messageReader.getScopeFilter(), messageReader.getDomain());
			if (dataSystems.size() > 1) {
				logger.warn("picking system to save but there are multiple possibilities " + dataSystems.size());
			}
			for (DataSystem possibleDataSystem : dataSystems) {
				dataSystem = possibleDataSystem;
			}
		}
		logger.debug("determined system to which to save data");
		return dataSystem;
	}

	private PatientIdentifiers translatePatientIdentifier(PatientIdentifier patientIdentifier, String targetAssigningAuthority) {
		PatientIdentifiers correspondingPatientIdentifiers = MhpObjectFactory.getInstance().getPatientCorrelationService().getCorrespondIdentifiers(patientIdentifier, targetAssigningAuthority);
		logTranslatedPatientIdentifiers(patientIdentifier, targetAssigningAuthority, correspondingPatientIdentifiers);
		return correspondingPatientIdentifiers;
	}

	private void logTranslatedPatientIdentifiers(PatientIdentifier originalPatientIdentifier, String targetAssigningAuthority, PatientIdentifiers correspondIdentifiers) {
		
		StringBuilder builder = new StringBuilder();
		for (PatientIdentifier correspondingIdentifier : correspondIdentifiers) {
			builder.append("[" + correspondingIdentifier + "]");
		}
		
		logger.debug("Translated patient identifier: "+targetAssigningAuthority);
	}

}
