package com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata;

import java.util.ArrayList;
import java.util.Collection;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestHandler;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDataCollection;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DataSystem;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DomainServiceRegistry;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.utility.NullChecker;

public class FetchListofPatientDataByScopeRequestHandler implements RequestHandler {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory
			.getLog(FetchListofPatientDataByScopeRequestHandler.class);

	@Override
	public ResponseMessage handle(RequestMessage parentRequestMessage) {
		logger.debug("handler received request to process message to retrieve patient data by scope");

		Collection<RequestMessage> childRequestMessages = expandParentRequestToChildRequests(parentRequestMessage);
		logger.debug("expanded request to retrieve patient data into multiple child requests");

		Collection<ResponseMessage> responses = executeChildRequestMessages(childRequestMessages);

		fixPatientIdentifier(responses);
		ResponseMessage parentResponseMessage = aggregate(responses);

		logger.debug("handler complete with processing request to process message to retrieve patient data by scope");
		return parentResponseMessage;
	}

	private int getResponseDataSize(ResponseMessage parentResponseMessage) {
		PatientDataResponseReader<PatientDataCollection<PatientData>, PatientData> parentResponseMessageReader = PatientDataResponseReader
				.<PatientDataCollection<PatientData>, PatientData> fromResponse(parentResponseMessage);
		int size = parentResponseMessageReader.getDataListSize();
		return size;
	}

	private void fixPatientIdentifier(Collection<ResponseMessage> responses) {
		for (ResponseMessage response : responses) {
			PatientDataResponseReader<?, ?> responseReader = PatientDataResponseReader.fromResponseWithoutType(response);
			if (responseReader.getDataList() != null) {
				responseReader.getDataList().updatePatientIdentifier(responseReader.getRequestReader().getPatientIdentifier());
				responseReader.getDataList().updateSystemId(responseReader.getRequestReader().getScopeFilter().getSystemIdentifier());
			}
		}
	}

	private ResponseMessage aggregate(Collection<ResponseMessage> responses) {
		PatientDataResponseBuilder parentResponseBuilder = PatientDataResponseBuilder.newInstance();

		for (ResponseMessage childResponse : responses) {
			PatientDataResponseReader<PatientDataCollection<PatientData>, PatientData> childResponseReader = PatientDataResponseReader
					.<PatientDataCollection<PatientData>, PatientData> fromResponse(childResponse);
			parentResponseBuilder.appendPatientData(childResponseReader.getDataList());
		}
		return parentResponseBuilder.build();
	}

	private MhpUser getCurrentUser(){
		return MhpUserFactory.createFromSecurityContext();
	}

	private Collection<RequestMessage> expandParentRequestToChildRequests(RequestMessage parentRequest) {
		PatientDataFetchRequestReader parentRequestReader = PatientDataFetchRequestReader.fromRequest(parentRequest);

		DomainServiceRegistry registry = new DomainServiceRegistry();
		Collection<DataSystem> datasystemList = registry.getDataSystems(parentRequestReader.getScopeFilter(), parentRequestReader.getDomain());

		Collection<RequestMessage> childRequestMessages = new ArrayList<RequestMessage>();

		logger.debug("begin expanding initial request into child requests, # of systems " + datasystemList.size());
		for (DataSystem targetDataSystem : datasystemList) {

			String targetSystemIdentifier = targetDataSystem.getSystemIdentifier();
			String targetAssigningAuthority = targetDataSystem.getPatientAssigningAuthority();

			if ("vista".equalsIgnoreCase(targetDataSystem.getSystemIdentifier())){
				MhpUser currentUser = getCurrentUser();
				targetAssigningAuthority = String.format(targetDataSystem.getPatientAssigningAuthority(),currentUser.getVistaLocation());
			}

			logger.debug("expanding initial request into child requests, checking for system");

			for (PatientIdentifier targetPatientIdentifier : translatePatientIdentifier(parentRequestReader.getPatientIdentifier(),
					targetAssigningAuthority)) {

				logger.debug("expanding initial request into child requests for system");

				PatientDataFetchRequestBuilder childRequestBuilder = PatientDataFetchRequestBuilder.cloneFromParent(parentRequest);
				childRequestBuilder.forPatientIdentifier(targetPatientIdentifier);
				childRequestBuilder.forTargetDataSystem(targetDataSystem);
				childRequestBuilder.forType(determineChildMessageType(parentRequest));
				childRequestMessages.add(childRequestBuilder.build());
			}
		}
		return childRequestMessages;
	}

	private String determineChildMessageType(RequestMessage parentRequest) {
		String childMessageType = parentRequest.getValue("childMessageType");
		if (NullChecker.isNullish(childMessageType)) {
			childMessageType = "{domain}.retrievelist";
		}
		PatientDataFetchRequestReader parentRequestReader = PatientDataFetchRequestReader.fromRequest(parentRequest);
		childMessageType = childMessageType.replace("{domain}", parentRequestReader.getDomain());

		return childMessageType;
	}

	private Collection<ResponseMessage> executeChildRequestMessages(Collection<RequestMessage> requestMessages) {
		Collection<ResponseMessage> responses = new ArrayList<ResponseMessage>();

		for (RequestMessage childRequest : requestMessages) {
			Router router = new Router();
			logger.debug("delegating child request to router");
			ResponseMessage childResponse = router.execute(childRequest);
			logger.debug("complete with child request");

			responses.add(childResponse);
		}

		return responses;
	}

	private PatientIdentifiers translatePatientIdentifier(PatientIdentifier patientIdentifier, String targetAssigningAuthority) {

		PatientIdentifiers matcingPatientIdentifiers = new PatientIdentifiers();

        try {
            if(targetAssigningAuthority.equals(patientIdentifier.getAssigningAuthority())){
                logger.debug("No translations are neccessary as patient's assigning authority and target assigning authority are same for the patient identifier s");
                matcingPatientIdentifiers.add(patientIdentifier);
                return matcingPatientIdentifiers;
            }

            // Get all the identifiers for given patient identifier
            PatientIdentifiers correspondingIdentifiers = MhpObjectFactory.getInstance().getPatientCorrelationService()
                    .getCorrespondIdentifiers(patientIdentifier);


            // Get all the identifiers matching the target assigning authority
            for (PatientIdentifier possibleMatch : correspondingIdentifiers) {
                if (possibleMatch.getAssigningAuthority().toLowerCase().contains(targetAssigningAuthority.toLowerCase())) {
                    matcingPatientIdentifiers.add(possibleMatch);
                }
            }

            logger.debug("translated patient identifier for assigning authority into corresponding identifiers");
        }
        catch(Exception e) {
            logger.error("Caught a " + e.getClass().getName() + ", returning empty object: " + e.toString());
        }

        return matcingPatientIdentifiers;
	}
}
