package com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata;

import java.util.Collection;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestHandler;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DataSystem;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DomainServiceRegistry;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.utility.NullChecker;

public class CreatePatientDataRequestHandler implements RequestHandler {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(CreatePatientDataRequestHandler.class);

	@Override
	public ResponseMessage handle(RequestMessage requestMessage) {
		logger.debug("received request");

		PatientDataEditRequestReader<PatientData> messageReader = PatientDataEditRequestReader.fromRequest(requestMessage);

		validateDataToCreate(messageReader);

		String childMessageType;
		if (requestMessage.getType().contentEquals("CreatePatientData")) {
			childMessageType = messageReader.getDomain() + ".create";
		} else {
			childMessageType = messageReader.getDomain() + requestMessage.getType();
		}

		// todo: handle 0 or many
		DataSystem system = determineSystem(messageReader);

		PatientIdentifier translatedPatientIdentifier = getTranslatedPatientIdentifier(messageReader, system);

		RequestMessage childRequestMessage = PatientDataEditRequestBuilder.cloneFromParent(requestMessage).forType(childMessageType).forData(messageReader.getData()).forPatientIdentifier(translatedPatientIdentifier).build();

		logger.debug("passing request for patient data to router to be routed to domain specific handler");
		Router router = new Router();
		ResponseMessage response = router.execute(childRequestMessage);

		fixDataIdentifier(response, system);
		fixPatientIdentifier(response, translatedPatientIdentifier);

		logger.debug("complete with execute request message");
		return response;
	}

	private void fixPatientIdentifier(ResponseMessage response, PatientIdentifier translatedPatientIdentifier) {
		PatientDataResponseReader<?, ?> responseReader = PatientDataResponseReader.fromResponseWithoutType(response);
		PatientIdentifier savedDataPatientIdentifier = responseReader.getDataItem().getPatientIdentifier();
		PatientIdentifier originalPatientIdentifier = PatientDataEditRequestReader.fromRequest( responseReader.getRequestReader().request).getData().getPatientIdentifier();
		
		
		logger.debug("checking patient identifier => " + savedDataPatientIdentifier);
		if ( NullChecker.isNullish(savedDataPatientIdentifier.getAssigningAuthority()) ) {
			responseReader.getDataItem().setPatientIdentifier(originalPatientIdentifier);
			logger.debug("fix patient identifier => " + responseReader.getDataItem().getPatientIdentifier());
		}

		// for (ResponseMessage response : responses) {
		// PatientDataResponseReader<?, ?> responseReader =
		// PatientDataResponseReader.fromResponseWithoutType(response);
		// responseReader.getDataList().updatePatientIdentifier(responseReader.getRequestReader().getPatientIdentifier());
		// responseReader.getDataList().updateSystemId(responseReader.getRequestReader().getScopeFilter().getSystemIdentifier());
		//
		// }

	}

	private void fixDataIdentifier(ResponseMessage response, DataSystem system) {
		PatientDataResponseReader<?, ?> responseReader = PatientDataResponseReader.fromResponseWithoutType(response);
		PatientData dataItem = responseReader.getDataItem();
		if (dataItem != null && NullChecker.isNullish(dataItem.getDataIdentifier().getSystemId())) {
			logger.debug("saved item");
			dataItem.getDataIdentifier().setSystemId(system.getSystemIdentifier());
			logger.debug("change saved item's dataid to reflect system/id");
		}
	}

	private void validateDataToCreate(PatientDataEditRequestReader<PatientData> messageReader) {
		logger.debug("validate request to save data");

		validateDataExists(messageReader);
		validatePatient(messageReader);
		validateScope(messageReader);

	}

	private void validateScope(PatientDataEditRequestReader<PatientData> messageReader) {
		if (messageReader.getScopeFilter() == null)
			throw new RuntimeException("Cannot create data without scope or system");
		if (NullChecker.isNullish(messageReader.getScopeFilter().getScopePreset()) && NullChecker.isNullish(messageReader.getScopeFilter().getSystemIdentifier()))
			throw new RuntimeException("Cannot create data without scope or system");
	}

	private void validatePatient(PatientDataEditRequestReader<PatientData> messageReader) {
		if (messageReader.getData().getPatientIdentifier() == null || NullChecker.isNullish(messageReader.getData().getPatientIdentifier().getUniqueId())
				|| NullChecker.isNullish(messageReader.getData().getPatientIdentifier().getAssigningAuthority()))
			throw new RuntimeException("Cannot create data without patient identifier");
	}

	private void validateDataExists(PatientDataEditRequestReader<PatientData> messageReader) {
		if (messageReader.getData() == null)
			throw new RuntimeException("Cannot create null data");
	}

	private PatientIdentifier getTranslatedPatientIdentifier(PatientDataEditRequestReader<PatientData> messageReader, DataSystem dataSystem) {
		PatientIdentifiers translatedPatientIdentifiers = getTranslatedPatientIdentifiers(messageReader, dataSystem);
		if (translatedPatientIdentifiers.size() == 0)
			throw new RuntimeException("Unable to translate patient identifier");
		return translatedPatientIdentifiers.get(0);
	}

	private PatientIdentifiers getTranslatedPatientIdentifiers(PatientDataEditRequestReader<PatientData> messageReader, DataSystem dataSystem) {
		PatientIdentifier patientIdentifier = messageReader.getData().getPatientIdentifier();

		PatientIdentifiers translatedPatientIdentifiers = null;
		if (dataSystem.getPatientAssigningAuthority().contentEquals(patientIdentifier.getAssigningAuthority())) {
			translatedPatientIdentifiers = new PatientIdentifiers();
			translatedPatientIdentifiers.add(patientIdentifier);
		} else {
			translatedPatientIdentifiers = translatePatientIdentifier(patientIdentifier, dataSystem.getPatientAssigningAuthority());
		}
		return translatedPatientIdentifiers;
	}

	private PatientIdentifiers translatePatientIdentifier(PatientIdentifier patientIdentifier, String targetAssigningAuthority) {
		PatientIdentifiers correspondingPatientIdentifiers = MhpObjectFactory.getInstance().getPatientCorrelationService().getCorrespondIdentifiers(patientIdentifier, targetAssigningAuthority);
		logTranslatedPatientIdentifiers(patientIdentifier, targetAssigningAuthority, correspondingPatientIdentifiers);
		return correspondingPatientIdentifiers;
	}

	private void logTranslatedPatientIdentifiers(PatientIdentifier originalPatientIdentifier, String targetAssigningAuthority, PatientIdentifiers correspondIdentifiers) {
		
		StringBuilder builder = new StringBuilder();
		for (PatientIdentifier correspondingIdentifier : correspondIdentifiers) {
			builder.append("[" + correspondingIdentifier + "]");
		}
		logger.debug("Translated patient identifier: "+targetAssigningAuthority);
	}

	private DataSystem determineSystem(PatientDataEditRequestReader<PatientData> messageReader) {
		DomainServiceRegistry registry = new DomainServiceRegistry();

		DataSystem dataSystem = null;
		DataIdentifier dataIdentifier = null;
		if (messageReader.getData() != null && messageReader.getData().getDataIdentifier() != null) {
			dataIdentifier = messageReader.getData().getDataIdentifier();
		}

		if (dataIdentifier != null && NullChecker.isNotNullish(dataIdentifier.getSystemId())) {
			dataSystem = registry.getDataSystemByName(messageReader.getData().getDataIdentifier().getSystemId());
		}

		if (dataSystem == null) {
			Collection<DataSystem> dataSystems = registry.getDataSystems(messageReader.getScopeFilter(), messageReader.getDomain());
			if (dataSystems.size() > 1) {
				logger.warn("picking system to save but there are multiple possibilities " + dataSystems.size());
			}
			for (DataSystem possibleDataSystem : dataSystems) {
				dataSystem = possibleDataSystem;
			}
		}
		logger.debug("determined system to which to save data");
		return dataSystem;

	}

}
