package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports;

import java.util.*;

import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.ReportGenerator;
import com.agilex.healthcare.mobilehealthplatform.datalayer.paingoal.PainGoalReportGenerator;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;
import org.apache.commons.collections.comparators.NullComparator;

public class PainConsolidatedReportGenerator {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ReportGenerator.class);

	private final Patient patient;
	private final Date startDate;
	private final Date endDate;
	private final AssessmentResults assessmentResults;
	private final PainDiaryEntries painDiaryEntries;
	private final PainGoals painGoals;
	private ToolTrackingResults toolTrackingResults;

	public PainConsolidatedReportGenerator(PainConsolidatedReportGeneratorParameter painConsolidatedReportGeneratorParameter) {
		this.assessmentResults = painConsolidatedReportGeneratorParameter.getAssessmentResults();
		this.painDiaryEntries = painConsolidatedReportGeneratorParameter.getPainDiaryEntries();
		this.painGoals = painConsolidatedReportGeneratorParameter.getPainGoals();
		this.toolTrackingResults = painConsolidatedReportGeneratorParameter.getToolTrackingResults();
		this.patient = painConsolidatedReportGeneratorParameter.getPatient();
		this.startDate = painConsolidatedReportGeneratorParameter.getStartDate();
		this.endDate = painConsolidatedReportGeneratorParameter.getEndDate();
	}

	public AssessmentResultReport generateReport() {
		AssessmentResultReport assessmentResultReport = new AssessmentResultReport();

		AssessmentProperties properties = new AssessmentProperties();
		properties.add(generateHeaderProperty());
		properties.add(generateSection1Property());
		properties.add(generateSection2Property());
		properties.add(generateSection3Property());
		properties.add(generateSection4Property());
		assessmentResultReport.setProperties(properties);
		return assessmentResultReport;
	}

	private AssessmentProperty generateHeaderProperty() {
        String header = HtmlReportBuilder.create("Summary Pain Report").bold().underline().center().build() +
                HtmlReportBuilder.create("Source: VA Pain Coach").center().build() +
                HtmlReportBuilder.create("Patient: " + this.patient.getDisplayName()).center().build() +
                HtmlReportBuilder.create("Period: " + DateHelper.formatDate(startDate) + " - " + DateHelper.formatDate(endDate)).center().nl().build();
        logger.debug("Pain Consolidated Report Header:");
        logger.debug(header);
		return new AssessmentProperty(AssessmentCode.PAIN_CONSOLIDATED_REPORT_HEADER, header);
	}

	private AssessmentProperty generateSection1Property() {
		String reportText = "";

		reportText += createSection1(assessmentResults);

		return new AssessmentProperty(AssessmentCode.PAIN_CONSOLIDATED_REPORT_MONTHLYPAINASSESSMENTS, reportText);
	}

	private AssessmentProperty generateSection2Property() {
		String reportText = "";

		reportText += createSection2(painDiaryEntries);
        logger.debug("Consolidated Report Pain Diary:");
        logger.debug(reportText);
		return new AssessmentProperty(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINDIARY, reportText);
	}

	private AssessmentProperty generateSection3Property() {
		String reportText = "";
		reportText += createSection3();

		return new AssessmentProperty(AssessmentCode.PAIN_CONSOLIDATED_REPORT_TOOLTRACKING, reportText);
	}

	private AssessmentProperty generateSection4Property() {

		PainGoalReportGenerator generator = new PainGoalReportGenerator(this.painGoals);
		PainGoalReport report = generator.generateReport();

		return new AssessmentProperty(AssessmentCode.PAIN_CONSOLIDATED_REPORT_PAINGOALS, report.getReport());
	}

    private String createSection1(AssessmentResults results){
        PainConsolidatedReportAssessmentGenerator painConsolidatedReportAssessmentGenerator = new PainConsolidatedReportAssessmentGenerator();
        return painConsolidatedReportAssessmentGenerator.createSection1(results, startDate, endDate);
    }

	private String createSection2(PainDiaryEntries painDiaryEntries) {
        String sectionText = HtmlReportBuilder.create("Pain Diary Report").bold().nl().build();
        Collections.sort(painDiaryEntries, new PainDiaryDateComparator());
        if (painDiaryEntries.size() >= 2){
            int days = DateHelper.calculateDeltaInDays(painDiaryEntries.get(0).getDate(), painDiaryEntries.get(painDiaryEntries.size()-1).getDate());

            sectionText += HtmlReportBuilder.create(String.format("During the last %1s Days (Average responses)", days)).bold().build();

        }

        int averageDailyPain = calculateAverageIntensity(painDiaryEntries);
        String averageResponses = HtmlReportBuilder.create("Daily Pain Intensity: " + averageDailyPain).listItem().build();
        averageResponses += HtmlReportBuilder.create("Depression Level: " + calculateAverageDepressionLevel()).listItem().build();
        averageResponses += HtmlReportBuilder.create("Pain interference on Movement: " + calculateAveragePainMovementImpact()).listItem().build();
        averageResponses += HtmlReportBuilder.create("Pain interference with Activities: " + calculateAveragePainInterference()).listItem().build();
        averageResponses += HtmlReportBuilder.create("Pain negative impact on Mood: " + calculateAverageMoodImpact()).listItem().build();
        averageResponses += HtmlReportBuilder.create("How refreshed from Sleep: " + calculateAverageSleepImpact()).listItem().build();
        sectionText += HtmlReportBuilder.create(averageResponses).unorderedList().nl().build();

        String quality = HtmlReportBuilder.create("Pain Quality/Descriptions (Top 5)").build();
        quality += HtmlReportBuilder.create(determineCommonQualityOfPain(painDiaryEntries)).unorderedList().build();
        sectionText += quality;

        String treatments = HtmlReportBuilder.create("Treatments Used (Top 5)").build();
        treatments += HtmlReportBuilder.create(determineMostUsedTreatments(painDiaryEntries)).unorderedList().build();
        sectionText += treatments;

        String activators = HtmlReportBuilder.create("Pain Activators (Last entry)").build();
        activators += HtmlReportBuilder.create(determineCommonPainActivators(painDiaryEntries)).unorderedList().build();
        sectionText += activators;

        String improved = HtmlReportBuilder.create("Items that Improved Pain (Last Entry)").build();
        improved += HtmlReportBuilder.create(determineItemsThatImprovedPain(painDiaryEntries)).unorderedList().build();
        sectionText += improved;

        String worsened = HtmlReportBuilder.create("Items that Worsened Pain (Last Entry)").build();
        worsened += HtmlReportBuilder.create(determineItemsThatWorsenPain(painDiaryEntries)).unorderedList().build();
        sectionText += worsened;

        String comments = HtmlReportBuilder.create("Comments").underline().build();
        comments += HtmlReportBuilder.create(determineComments()).unorderedList().build();
        sectionText += comments;

		return sectionText; // To change body of created methods use File |
							// Settings | File Templates.
	}

    private String determineCommonPainActivators(PainDiaryEntries painDiaryEntries) {
        return new RecentActivatorsListItemsBuilder(painDiaryEntries).build();
	}

	private String determineItemsThatImprovedPain(PainDiaryEntries painDiaryEntries) {
        return new RecentItemsThatImproveListItemsBuilder(painDiaryEntries).build();
	}

	private String determineItemsThatWorsenPain(PainDiaryEntries painDiaryEntries) {
        return new RecentItemsThatWorsenListItemsBuilder(painDiaryEntries).build();
	}

	private String determineComments() {
		String comments = "";
		for (PainDiaryEntry entry : painDiaryEntries) {
            String painDiaryComment = entry.getPainQuestionValues().getComments();
            if (painDiaryComment != null){
                String comment = String.format("%1s: %2s", DateHelper.formatDate(entry.getDateCreated()), entry.getPainQuestionValues().getComments());
                comments += HtmlReportBuilder.create(comment).listItem().build();
            }
		}
		return comments;
	}

	private int calculateAverageIntensity(PainDiaryEntries painDiaryEntries) {
		int total = 0;
		int count = 0;

		for (PainDiaryEntry entry : painDiaryEntries) {
			String painValue = entry.getPainQuestionValues().getAveragePainIntensity();
			if ((null != painValue) && (!"".equals(painValue))) {
				total += Integer.parseInt(painValue);
			}
			count++;
		}

		if (count != 0) {
			return total / count; // To change body of created methods use File
									// | Settings | File Templates.
		} else {
			return 0;
		}
	}

	private int calculateAveragePainMovementImpact() {
		int total = 0;
		int count = 0;

		for (PainDiaryEntry entry : painDiaryEntries) {
			String value = entry.getPainQuestionValues().getImpactOnWalking();
			if ((null != value) && (!"".equals(value))) {
				total += Integer.parseInt(value);
			}
			count++;
		}

		if (count != 0) {
			return total / count; // To change body of created methods use File
									// | Settings | File Templates.
		} else {
			return 0;
		}
	}

	private int calculateAverageSleepImpact() {
		int total = 0;
		int count = 0;

		for (PainDiaryEntry entry : painDiaryEntries) {
			String value = entry.getPainQuestionValues().getAmountRefreshedAfterSleep();
			if ((null != value) && (!"".equals(value))) {
				total += Integer.parseInt(value);
			}
			count++;
		}

		if (count != 0) {
			return total / count; // To change body of created methods use File
									// | Settings | File Templates.
		} else {
			return 0;
		}
	}

	private int calculateAverageMoodImpact() {
		int total = 0;
		int count = 0;

		for (PainDiaryEntry entry : painDiaryEntries) {
			String value = entry.getPainQuestionValues().getMoodLevel();
			if ((null != value) && (!"".equals(value))) {
				total += Integer.parseInt(value);
			}
			count++;
		}

		if (count != 0) {
			return total / count; // To change body of created methods use File
									// | Settings | File Templates.
		} else {
			return 0;
		}
	}

	private int calculateAveragePainInterference() {
		int total = 0;
		int count = 0;

		for (PainDiaryEntry entry : painDiaryEntries) {
			String value = entry.getPainQuestionValues().getPainInterference();
			if ((null != value) && (!"".equals(value))) {
				total += Integer.parseInt(value);
			}
			count++;
		}

		if (count != 0) {
			return total / count; // To change body of created methods use File
									// | Settings | File Templates.
		} else {
			return 0;
		}
	}

	private int calculateAverageDepressionLevel() {
		int total = 0;
		int count = 0;

		for (PainDiaryEntry entry : painDiaryEntries) {
			String value = entry.getPainQuestionValues().getDepressionLevel();
			if ((null != value) && (!"".equals(value))) {
				total += Integer.parseInt(value);
			}
			count++;
		}

		if (count != 0) {
			return total / count; // To change body of created methods use File
									// | Settings | File Templates.
		} else {
			return 0;
		}
	}

	private String determineCommonQualityOfPain(PainDiaryEntries painDiaryEntries) {
        List<String> qualityOfPainList = buildListOfPainQuality(painDiaryEntries);
        List<String> top5Entries = determineTopOccurances(qualityOfPainList, 5);
        return emitListAsListItems(top5Entries);
	}

    private String determineMostUsedTreatments(PainDiaryEntries painDiaryEntries) {
        List<String> qualityOfPainList = buildListOfTreatments(painDiaryEntries);
        List<String> top5Entries = determineTopOccurances(qualityOfPainList, 5);
        return emitListAsListItems(top5Entries);
    }

    private String emitListAsListItems(List<String> entries) {
        String commonQuality = HtmlReportBuilder.create("").build();
        for (String s : entries) {
            commonQuality += HtmlReportBuilder.create(s).listItem().build();
        }
        return commonQuality;
    }

    private List<String> buildListOfPainQuality(PainDiaryEntries painDiaryEntries) {
        //build list of all quality of pain entries
        List<String> qualityOfPainList = new ArrayList<String>();
        for (PainDiaryEntry painDiaryEntry : painDiaryEntries) {
            PainDiaryReport report = painDiaryEntry.getPainQuestionValues();
            if (report != null){
                List<String> painQuality = report.getPainQuality();
                if (painQuality != null){
                    for (String s : painQuality) {
                        qualityOfPainList.add(s);
                    }
                }
            }
        }
        return qualityOfPainList;
    }

    private List<String> buildListOfTreatments(PainDiaryEntries painDiaryEntries) {
        //build list of all quality of pain entries
        List<String> treatmentList = new ArrayList<String>();
        for (PainDiaryEntry painDiaryEntry : painDiaryEntries) {
            PainDiaryReport report = painDiaryEntry.getPainQuestionValues();
            if (report != null){
                List<String> treatments = report.getTreatment();
                if (treatments != null){
                    for (String s : treatments) {
                        treatmentList.add(s);
                    }
                }
            }
        }
        return treatmentList;
    }

    private List<String> determineTopOccurances(List<String> qualityOfPainList, int numberOfEntriesDesired) {
        //build counts
        Map<String, EntryCount> qualityOfPainCountMap = new HashMap<String, EntryCount>();
        for (String s : qualityOfPainList) {
            //either add to the map, or increment if it already exists in map
            if (qualityOfPainCountMap.containsKey(s)){
                qualityOfPainCountMap.get(s).count += 1;
            }
            else{
                EntryCount painCount = new EntryCount();
                painCount.count = 1;
                painCount.qualityOfPain = s;
                qualityOfPainCountMap.put(s, painCount);
            }
        }

        List<EntryCount> painCountArrayList = new ArrayList<EntryCount>(qualityOfPainCountMap.values());
        Collections.sort(painCountArrayList, new QualityOfPainCountReverseSortyByCount());

        //limit to top 5 entries
        List<String> topEntries = new ArrayList<String>();
        for (EntryCount painCount : painCountArrayList) {
            topEntries.add(painCount.qualityOfPain);
            if (topEntries.size() == numberOfEntriesDesired){
                break;
            }
        }
        return topEntries;
    }

	private String createSection3() {
		String report = HtmlReportBuilder.create("").nl().build();
        report += HtmlReportBuilder.create("Tool Tracking Summary").bold().nl().build();
        report += HtmlReportBuilder.create("Most recently reported data").nl().build();
        report += HtmlReportBuilder.create(String.format("From: %1s to %2s", DateHelper.formatDate(startDate), DateHelper.formatDate(endDate))).nl().nl().build();

		if (toolTrackingResults != null) {

            //Reverse chronological order to determine which tool results to emit
			Collections.sort(toolTrackingResults, new ToolTrackingReverseDateComparator());
            Map<String, ToolTrackingResult> foundTools = new TreeMap<String, ToolTrackingResult>();   //use a tree map so it gets sorted by the key/tool name

            report += HtmlReportBuilder.create("<table> <colgroup> <col> <col style=\"width: 10px\"> <col> <col style=\"width: 10px\"> <col> </colgroup>").build();
            String headerlineFormat =
                    "<tr style='font-weight: bold;vertical-align: top'>" +
                        "<td>%1$s</td>" +
                        "<td width=\"10px\"></td>" +
                        "<td style=\"vertical-align: top\"><div  style=\"text-align: center\">%2$s</div></td>" +
                        "<td width=\"10px\"></td>" +
                        "<td style=\"vertical-align: top\"><div  style=\"text-align: center\">%3$s</div></td>" +
                        "</tr>";
            String lineFormat = "<tr>" +
                    "<td style=\"text-align: left\">%1$s</td>" +
                    "<td width=\"10px\"></td>" +
                    "<td style=\"text-align: center\">%2$s</td>" +
                    "<td width=\"10px\"></td>" +
                    "<td style=\"text-align: center\">%3$s</td>" +
                    "</tr>";
            String header = HtmlReportBuilder.create(String.format(headerlineFormat, "Tool Name", "Success<div style=\"text-align: center\">Rate</div>", "Usefulness<div style=\"text-align: center\">(0 - 10)</div>")).build();
            report += header;

            //only get one result for each tool
			for (ToolTrackingResult toolTrackingResult : toolTrackingResults) {
                String tool = toolTrackingResult.getTool();
                //only emit a tool once
                if ((tool != null) && !foundTools.containsKey(tool)){
                    foundTools.put(tool, toolTrackingResult);  //add tool to list so we don't emit it again
                }
			}

            //emit in alphabetical order (sorted automatically by the TreeMap
            for (ToolTrackingResult toolTrackingResult : foundTools.values()){
                Integer successrate = toolTrackingResult.getSuccessRate();
                String successrateText = "";
                if (successrate != null){
                    successrateText = successrate.toString() + "%";
                }

                Integer usefulness = toolTrackingResult.getUsefulness();
                String usefulnessText = "";
                if (usefulness != null){
                    usefulnessText = usefulness.toString();
                }
                String line = HtmlReportBuilder.create(String.format(lineFormat, toolTrackingResult.getTool(), successrateText, usefulnessText)).build();
                report += "\n" + line;
            }

            report += HtmlReportBuilder.create("</table>").build();
		}

		return report; 
	}
    class ToolTrackingReverseDateComparator implements Comparator<ToolTrackingResult> {

        @Override
        public int compare(ToolTrackingResult toolTrackingResult, ToolTrackingResult toolTrackingResult1) {
            NullComparator nullComparator = new NullComparator();
            int result = nullComparator.compare(toolTrackingResult.getDate(), toolTrackingResult1.getDate());
            return result * -1; // reverse order
        }
    }

    class PainDiaryDateComparator implements Comparator<PainDiaryEntry> {

        @Override
        public int compare(PainDiaryEntry painDiaryEntry, PainDiaryEntry painDiaryEntry1) {
            return painDiaryEntry.getDate().compareTo(painDiaryEntry1.getDate()); // reverse order
        }
    }

    class EntryCount {
        String qualityOfPain;
        Integer count;
    }

    class QualityOfPainCountReverseSortyByCount implements  Comparator<EntryCount>{

        @Override
        public int compare(EntryCount entryCount, EntryCount entryCount1) {
            return -1 * entryCount.count.compareTo(entryCount1.count);
        }
    }
}

