package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports;

import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.ReportGenerator;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;

import java.util.*;

public class PainConsolidatedReportAssessmentGenerator {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ReportGenerator.class);

	private int getMonth(Date givenDate) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(givenDate);
		int month = cal.get(Calendar.MONTH);
		return month;
	}

	private int getYear(Date givenDate) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(givenDate);
		int year = cal.get(Calendar.YEAR);
		return year;
	}

	private boolean needToAddMonthEntry(Date prevDate, Date currDate) {
		int prevMonth = getMonth(prevDate);
		int currMonth = getMonth(currDate);

		if ((currMonth == prevMonth) || (currMonth == prevMonth + 1) || (prevMonth == 11 && currMonth == 0)) {
			return false;
		} else {
			return true;
		}
	}

	public static Date getNextMonth(Date givenDate) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(givenDate);

		cal.add(Calendar.MONTH, 1);
		Date nextMonth = cal.getTime();

		return nextMonth;
	}

	private void addEntriesForMonthsBetween(AssessmentResults results) {
		List<AssessmentResult> entriesToAdd = new ArrayList<AssessmentResult>();

		AssessmentResult prevResult = null;
		for (AssessmentResult result : results) {
			if (prevResult == null) {
				prevResult = result;
			} else if (needToAddMonthEntry(prevResult.getDateTaken(), result.getDateTaken())) {
				while (needToAddMonthEntry(prevResult.getDateTaken(), result.getDateTaken())) {
					AssessmentResult newResult = createStubbedAssessmentResult(prevResult);
					entriesToAdd.add(newResult);
					prevResult = newResult;
				}
			} else {
				prevResult = result;
			}
		}

		for (AssessmentResult assessmentResult : entriesToAdd) {
			results.add(assessmentResult);
		}
	}

	private void removeMultipleEntiesPerMonth(AssessmentResults results) {
		List<AssessmentResult> entriesToRemove = new ArrayList<AssessmentResult>();

		AssessmentResult prevResult = null;
		for (AssessmentResult result : results) {
			if (prevResult == null) {
				prevResult = result;
			} else {
				Date prevDate = prevResult.getDateTaken();
				Date currDate = result.getDateTaken();
				if ((getMonth(prevDate) == getMonth(currDate)) && (getYear(prevDate) == getYear(currDate))) {
					entriesToRemove.add(prevResult);
				}
				prevResult = result;
			}
		}

		// To change body of created methods use File | Settings | File
		// Templates.
		for (AssessmentResult assessmentResult : entriesToRemove) {
			results.remove(assessmentResult);
		}
	}

	private AssessmentResult createStubbedAssessmentResult(AssessmentResult prevResult) {
		AssessmentResult newResult = new AssessmentResult();
		Date newDateTaken = getNextMonth(prevResult.getDateTaken());
		newResult.setDateTaken(newDateTaken);

		AssessmentResponses responses = new AssessmentResponses();
		AssessmentQuestionChoices choices = new AssessmentQuestionChoices();
		AssessmentQuestionChoice assessmentQuestionChoice = new AssessmentQuestionChoice();
		AssessmentProperties assessmentProperties = new AssessmentProperties();
		AssessmentProperty assessmentProperty = new AssessmentProperty();
		assessmentProperty.setName(AssessmentCode.RESULT_PROPERTY_SELECTED_VALUE);
		assessmentProperty.setValue("");
		assessmentProperties.add(assessmentProperty);
		assessmentQuestionChoice.setProperties(assessmentProperties);
		choices.add(assessmentQuestionChoice);

		AssessmentResponse response = new AssessmentResponse();
		AssessmentQuestion assessmentQuestion = new AssessmentQuestion();
		response.setQuestion(assessmentQuestion);
		response.setSelectedChoices(choices);

		responses.add(response);
		newResult.setResponses(responses);

		return newResult;
	}

	public String createSection1(AssessmentResults results, Date startDate, Date endDate) {
		// Sort results
		Collections.sort(results, new AssessmentDateComparator());

		// remove multiple entries per Month
		removeMultipleEntiesPerMonth(results);

		// Make sure we have entries (even if blank) for every month from start
		// to end
		addEntriesForMonthsBetween(results);

		// Re-Sort results since new ones may have been added above
		Collections.sort(results, new AssessmentDateComparator());

		// Summary of Your Monthly Pain Assessments
		// Display: Table of all Pain assessment entries over the timeframe
		String sectionText = "";

		sectionText += HtmlReportBuilder.create("Monthly Pain Assessment Report").bold().nl().build();
		sectionText += HtmlReportBuilder.create(String.format("Showing data from %1s to %2s", DateHelper.formatDate(startDate), DateHelper.formatDate(endDate))).nl().nl().build();

		String[] questionList = { "Q1", "Q2", "Q3", "Q4", "Q5", "Q6", "Q7", "Q8", "Q9", "Q10", "Q11", "Q12", "Q13" };
		Map<String, List<QuestionResult>> questionResultsMap = new HashMap<String, List<QuestionResult>>();

		// build the results by question id
		for (String reportQuestionId : questionList) {
			List<QuestionResult> questionResults = new ArrayList<QuestionResult>();

			// get responses for this question

			for (AssessmentResult result : results) {
				questionResults.add(getResultQuestionResponse(reportQuestionId, result));
			}
			questionResultsMap.put(reportQuestionId, questionResults);
		}

		// emit
		for (String reportQuestionId : questionList) {
			sectionText += HtmlReportBuilder
					.create(String.format("<div style='background:#ffffff;border:solid black 1px; padding: 10px'>%1s: %2s", reportQuestionId, PainConsolidatedReportHelper.getMonthlyPainAssessmentQuestionDescription(reportQuestionId))).nl().build();
			sectionText += HtmlReportBuilder.create("<table><tr style=\"background-color: #4f81bd;\">").build();

			// get results for that question
			List<QuestionResult> questionResults = questionResultsMap.get(reportQuestionId);

			for (QuestionResult questionResult : questionResults) {
				sectionText += HtmlReportBuilder.create(String.format("<td style=\"color: #ffffff;\">%1s</td>", formatDateForResponse(questionResult)));
			}
			sectionText += HtmlReportBuilder.create("</tr><tr style=\"background-color: #d0d8e9\">").build();
			for (QuestionResult questionResult : questionResults) {
				sectionText += HtmlReportBuilder.create(String.format("<td>%1s</td>", questionResult.response));
			}
			sectionText += HtmlReportBuilder.create("</tr></table></div>").nl().build();
		}

		logger.debug(sectionText);
		return sectionText;
	}

	private String formatDateForResponse(QuestionResult questionResult) {
		Date resultDate = questionResult.dateTaken;
		if (resultDate != null) {
			return DateHelper.format(resultDate, "MMM");
		} else {
			return "";
		}
	}

	private QuestionResult getResultQuestionResponse(String reportQuestionId, AssessmentResult result) {
		for (AssessmentResponse assessmentResponse : result.getResponses()) {
			AssessmentQuestion question = assessmentResponse.getQuestion();
			String questionId = question.getId();

			if (reportQuestionId.equals(questionId)) {

				for (AssessmentQuestionChoice choice : assessmentResponse.getSelectedChoices()) {
					String value = choice.getProperties().getValueByPropertyName(AssessmentCode.RESULT_PROPERTY_SELECTED_VALUE);
					// we found a response we're looking for, copy needed info
					QuestionResult questionResult = new QuestionResult();
					questionResult.dateTaken = result.getDateTaken();
					questionResult.response = value;
					return questionResult;
				}
			}
		}

		// Didn't find it, return blank entry
		QuestionResult questionResult = new QuestionResult();
		questionResult.dateTaken = result.getDateTaken();
		questionResult.response = "";
		return questionResult;
	}

	private class AssessmentDateComparator implements Comparator<AssessmentResult> {

		@Override
		public int compare(AssessmentResult assessmentResult, AssessmentResult assessmentResult1) {
			int result = assessmentResult.getDateTaken().compareTo(assessmentResult1.getDateTaken());
			return result;
		}
	}

	private class QuestionResult {
		public Date dateTaken;
		public String response;
	}

}
