package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperties;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperty;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponse;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponses;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResultReport;
import com.agilex.healthcare.mobilehealthplatform.domain.ToolCategories;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;

public class ZaritReportGenerator implements ReportGenerator {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PTSDReportGenerator.class);
	private AssessmentResult currentResult = null;
	private AssessmentResult previousResult = null;
	private SummationScorer scorer = new SummationScorer();
	private ZaritReportSettings reportSettings = new ZaritReportSettings();
	private ZaritToolCalculator toolCalculator = new ZaritToolCalculator();

	public ZaritReportGenerator() {
	}

	public AssessmentResultReport generateReport(AssessmentResult currentAssessmentResult, AssessmentResult previousAssessmentResult) {
		AssessmentResultReport assessmentResultReport = new AssessmentResultReport();

		this.currentResult = currentAssessmentResult;
		this.previousResult = previousAssessmentResult;

		AssessmentResponses responsesOverCutoff = determineResponsesOverCutoff();

		ToolCategories toolCategories = generateToolsList(responsesOverCutoff);
		assessmentResultReport.setToolCategories(toolCategories);

		AssessmentProperties properties = generateProperties(responsesOverCutoff);
		assessmentResultReport.setProperties(properties);

		return assessmentResultReport;
	}

	private AssessmentProperties generateProperties(AssessmentResponses responsesOverCutoff) {
		AssessmentProperties properties = new AssessmentProperties();
		properties.add(generateSummaryTextProperty());

		if (responsesOverCutoff.size() > 0) {
			properties.add(generateHighScoringQuestionsExplanationProperty());
		}
		return properties;
	}

	private AssessmentProperty generateHighScoringQuestionsExplanationProperty() {
		AssessmentProperty explanationProperty = new AssessmentProperty(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION,
				reportSettings.highScoringQuestionsExplanation);
		return explanationProperty;
	}

	private ToolCategories generateToolsList(AssessmentResponses responsesOverCutoff) {
		ToolCategories toolsList = new ToolCategories();
		if (responsesOverCutoff != null) {
			int[] responsesOverCutoffArray = new int[responsesOverCutoff.size()];
			int i = 0;
			for (AssessmentResponse assessmentResponse : responsesOverCutoff) {
				responsesOverCutoffArray[i] = assessmentResponse.getQuestion().getQuestionNumber();
				i++;
			}
			toolsList = toolCalculator.getRecommendedToolCategories(responsesOverCutoffArray);
		}
		return toolsList;
	}

	private AssessmentResponses determineResponsesOverCutoff() {
		AssessmentResponses responsesOverCutoff = new AssessmentResponses();
		AssessmentResponses currentResultResponses = currentResult.getResponses();
		int numberOfResponses = currentResultResponses.getSize();

		for (int i = 0; i < numberOfResponses; i++) {
			AssessmentResponse currentResponse = currentResultResponses.get(i);

			int currentResponseScore = scorer.calculateResponsePointValue(currentResponse);
			boolean currentScoreIsOverCutoff = currentResponseScore >= reportSettings.cuttoffResponseScore;
			boolean deltaOverCutoff = checkIfResponseDeltaOverCutoff(currentResponse, i, currentResponseScore);

			if (currentScoreIsOverCutoff || deltaOverCutoff) {
				responsesOverCutoff.add(currentResponse);
			}
		}
		return responsesOverCutoff;
	}

	private boolean checkIfResponseDeltaOverCutoff(AssessmentResponse response, int i, int currentResponseScore) {
		if (previousResult != null && previousResult.getResponses().size() == currentResult.getResponses().size()) {
			AssessmentResponses allPreviousResponses = previousResult.getResponses();
			AssessmentResponse previousResponse = allPreviousResponses.get(i);
			int previousResponseScore = scorer.calculateResponsePointValue(previousResponse);
			int deltaResponseScore = currentResponseScore - previousResponseScore;
			if (deltaResponseScore > reportSettings.cutoffResponseScoreDelta) {
				return true;
			}
		} else if (previousResult != null && previousResult.getResponses().size() != currentResult.getResponses().size()) {
			logger.debug("current result and previous result are different sizes");
		}
		return false;
	}

	public AssessmentProperty generateSummaryTextProperty() {

		String resultsSummary = String.format("Your score on %1s was %d out of %d. \n\n", DateHelper.formatDateTimeInReportFormat(DateHelper.getToday()), currentResult.getScore(), reportSettings.maxScore);

		if (previousResult != null) {
			int deltaTotalScore = currentResult.getScore() - previousResult.getScore();
			Date previousDateTaken = previousResult.getDateTaken();
			String formattedPreviousDate = DateHelper.formatDateTimeInReportFormat(previousDateTaken);

			if (deltaTotalScore > 0) {
				resultsSummary += String.format("This is an increase of %d since your last assessment on " + formattedPreviousDate + ".\n\n", deltaTotalScore);
			}

			if (deltaTotalScore < 0) {
				resultsSummary += String.format("This is a decrease of %d since your last assessment on " + formattedPreviousDate + ".\n\n", -deltaTotalScore);
			}
			
			if (deltaTotalScore == 0) {
				resultsSummary += String.format("This is no change compared to your last assessment on " + formattedPreviousDate + ".\n\n");
			}

			if (deltaTotalScore > reportSettings.cutoffResponseScoreDelta) {
				resultsSummary += reportSettings.significantIncreaseText;
			}
		}

		int totalScore = currentResult.getScore();
		if (totalScore == 0) {
			resultsSummary += reportSettings.zeroScoreSummaryText;
		} else if (totalScore <= reportSettings.cutoffTotalScoreLow) {
			resultsSummary += reportSettings.lowScoreSummaryText;
		} else if (totalScore <= reportSettings.cutoffTotalScoreHigh) {
			resultsSummary += reportSettings.mediumScoreSummaryText;
		} else {
			resultsSummary += reportSettings.highScoreSummaryText;
		}

		return new AssessmentProperty(AssessmentCode.REPORT_SUMMARY, resultsSummary);
	}
}
