package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;

public class PTSDReportGenerator implements ReportGenerator {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ReportGenerator.class);

	private AssessmentResult currentResult = null;
	private AssessmentResult previousResult = null;
	private SummationScorer scorer = new SummationScorer();
	private PTSDReportSettings reportSettings = new PTSDReportSettings();
	private PTSDToolCalculator toolCalculator = new PTSDToolCalculator();
	private boolean patientHasAppointmentWithinTwoWeeks = false;

	public PTSDReportGenerator() {
	}

	public AssessmentResultReport generateReport(AssessmentResult currentAssessmentResult, AssessmentResult previousAssessmentResult) {
		this.currentResult = currentAssessmentResult;
		this.previousResult = previousAssessmentResult;

		AssessmentResultsDataService dataservice = new AssessmentResultsDataService();
		patientHasAppointmentWithinTwoWeeks = dataservice.checkPatientHasAppointmentWithinTwoWeeks(currentAssessmentResult.getPatientIdentifier());

		logger.debug("generating PTSD report");
		return generateReport();
	}

	private AssessmentResultReport generateReport() {
		AssessmentResultReport assessmentResultReport = new AssessmentResultReport();

		AssessmentResponses responsesOverCutoff = determineResponsesOverCutoff();
		ToolCategories toolCategories = generateToolsList(responsesOverCutoff);
		assessmentResultReport.setToolCategories(toolCategories);

		AssessmentProperties properties = new AssessmentProperties();
		properties.add(generateSummaryTextProperty());

		addToolsExplanationIfNeeded(responsesOverCutoff.size() > 0, properties);
		assessmentResultReport.setProperties(properties);
		return assessmentResultReport;
	}

	private void addToolsExplanationIfNeeded(boolean recommendedToolsExist, AssessmentProperties properties) {
		if (recommendedToolsExist) {
			AssessmentProperty explanationProperty = new AssessmentProperty(AssessmentCode.REPORT_HIGH_SCORING_QUESTIONS_EXPLANATION, reportSettings.highScoringQuestionsExplanation);
			properties.add(explanationProperty);
		}
	}

	private ToolCategories generateToolsList(AssessmentResponses responsesOverCutoff) {
		ToolCategories toolsList = new ToolCategories();
		if (responsesOverCutoff != null) {
			int[] responsesOverCutoffArray = new int[responsesOverCutoff.size()];
			int i = 0;
			for (AssessmentResponse assessmentResponse : responsesOverCutoff) {
				responsesOverCutoffArray[i] = assessmentResponse.getQuestion().getQuestionNumber();
				i++;
			}
			toolsList = toolCalculator.getRecommendedToolCategories(responsesOverCutoffArray);
		}
		return toolsList;
	}

	private AssessmentResponses determineResponsesOverCutoff() {
		AssessmentResponses responsesOverCutoff = new AssessmentResponses();
		AssessmentResponses currentResultResponses = currentResult.getResponses();

		for (int i = 0; i < currentResultResponses.getSize(); i++) {
			AssessmentResponse currentResponse = currentResultResponses.get(i);
			int currentResponseScore = scorer.calculateResponsePointValue(currentResponse);
			if (currentResponseScore >= reportSettings.cuttoffResponseScore) {
				responsesOverCutoff.add(currentResponse);
			}
		}
		return responsesOverCutoff;
	}

	private AssessmentProperty generateSummaryTextProperty() {

		String resultsSummary = String.format("Your score on %1s was %d out of %d. \n\n", DateHelper.formatDateTimeInReportFormat(DateHelper.getToday()), currentResult.getScore(), reportSettings.maxScore);

		if (previousResult == null) {
			resultsSummary += reportSettings.noPreviousResultText;
		} else {
			logger.debug(String.format("calculating delta in score [prev score=%s][current score=%s]", /*previousResult.getDataIdentifier(),*/ previousResult.getScore(), currentResult.getScore()));

			int deltaTotalScore = currentResult.getScore() - previousResult.getScore();
			if (deltaTotalScore == 0) {
				resultsSummary += reportSettings.sameScoreText;
			} else if (deltaTotalScore > 0) {
				resultsSummary += reportSettings.increasedScoreText;
			} else if (deltaTotalScore < 0) {
				resultsSummary += reportSettings.decreasedScoreText;
			}
		}

		int totalScore = currentResult.getScore();
		if (totalScore == reportSettings.lowestScoreCutoff) {
			resultsSummary += reportSettings.lowestScoreText;
		} else {
			if (totalScore <= reportSettings.lowScoreCutoff) {
				resultsSummary += reportSettings.lowScoreText;
			} else if (totalScore <= reportSettings.mediumScoreCutoff) {
				resultsSummary += reportSettings.mediumScoreText;
			} else {
				resultsSummary += reportSettings.highScoreText;
			}
		}

		if (totalScore > reportSettings.mediumScoreCutoff && !patientHasAppointmentWithinTwoWeeks) {
			resultsSummary += reportSettings.highScoreAndNotScheduledToSeeProviderText;
		} else if (patientHasAppointmentWithinTwoWeeks) {
			resultsSummary += reportSettings.scheduledToSeeProviderText;
		} else {
			resultsSummary += reportSettings.notScheduledToSeeProviderText;
		}

		resultsSummary += reportSettings.endOfAllResultsText;

		return new AssessmentProperty(AssessmentCode.REPORT_SUMMARY, resultsSummary);

	}
}