package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import java.util.ArrayList;
import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports.HtmlReportBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports.PainConsolidatedReportHelper;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperties;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperty;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestion;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponse;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponses;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResultReport;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.IntegerParser;

public class MonthlyPainReportGenerator implements ReportGenerator {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ReportGenerator.class);

	private AssessmentResult currentResult = null;
	private AssessmentResult previousResult = null;
	private MonthlyPainReportSettings reportSettings = new MonthlyPainReportSettings();

	public MonthlyPainReportGenerator() {
	}

	public AssessmentResultReport generateReport(AssessmentResult currentAssessmentResult, AssessmentResult previousAssessmentResult) {
		this.currentResult = currentAssessmentResult;
		this.previousResult = previousAssessmentResult;

		return generateReport();
	}

	private AssessmentResultReport generateReport() {
		AssessmentResultReport assessmentResultReport = new AssessmentResultReport();

		AssessmentProperties properties = new AssessmentProperties();
		properties.add(generateSummaryTextProperty());

		assessmentResultReport.setProperties(properties);
		return assessmentResultReport;
	}

	
	private void sortResponses(AssessmentResponses unsortedResponses){
		unsortedResponses.sortAscending("question.questionNumber");
	}
	
	private AssessmentProperty generateSummaryTextProperty() {

		Date submittedDate = currentResult.getDateTaken();
		String resultsSummary = "\n";

        resultsSummary += HtmlReportBuilder.create(reportSettings.title).center().bold().build();
        resultsSummary += HtmlReportBuilder.create(String.format("Date: %1s",  DateHelper.formatDate(submittedDate))).center().nl().build();

		ArrayList<String> improvedResults = new ArrayList<String>();
		ArrayList<String> noChangeResults = new ArrayList<String>();
		ArrayList<String> worsenedResults = new ArrayList<String>();

		if (previousResult == null){
			resultsSummary += HtmlReportBuilder.create(reportSettings.noPreviousResultText).nl().build();
		}
		else{
			resultsSummary += HtmlReportBuilder.create(String.format(reportSettings.changeHeaderText, DateHelper.formatDate(previousResult.getDateTaken()))).nl().build();
		}

		//Sort Responses
		sortResponses(currentResult.getResponses());
		
		for (AssessmentResponse response : currentResult.getResponses()) {
			AssessmentQuestion question = response.getQuestion();
			String questionId = question.getId();
			
			logger.debug("CurrentResutl QuestionId ");
			
			if (previousResult == null){

				String value = getResponseSelectedValue(response);
				String questionDescription = HtmlReportBuilder.create(String.format("%1s: %1s Score: %1s\n\n",  questionId, question.getPrompt(), value)).nl().build();
				resultsSummary += questionDescription;
			}
			else{
				logger.debug("calculating delta in score");
				//find matching question in prev result
				for (AssessmentResponse prevResponse : previousResult.getResponses()) {
					if (questionId.equals(prevResponse.getQuestion().getId())){

						String value = getResponseSelectedValue(response);
						String prevValue = getResponseSelectedValue(prevResponse);

                        String qD1 = HtmlReportBuilder.create(String.format("%1s: %1s",  questionId, getQuestionDescription(questionId))).build();

						logger.debug("Comparing Questions: " + questionId);
						//Don't try to compute anything if we end up with null value somehow
						Integer intValue = IntegerParser.parseInteger(value,  null);
						Integer intPrevValue = IntegerParser.parseInteger(prevValue, null); 
						
						if (null != intValue && null != intPrevValue){							
							int valueChange = intValue - intPrevValue;
							
							if (valueChange >= 1) {
                                String qD2 = HtmlReportBuilder.create(String.format("Increased from %1s to %1s",  prevValue, value)).nl().build();
                                String questionDescription = HtmlReportBuilder.create(qD1).nl().build() + qD2;
                                addIncreasedScoreResults(improvedResults, questionId, questionDescription, improvedResults, worsenedResults, noChangeResults);
                            }
							else if (valueChange <= -1){
                                String qD2 = HtmlReportBuilder.create(String.format("Decreased from %1s to %1s",  prevValue, value)).nl().build();
                                String questionDescription = HtmlReportBuilder.create(qD1).nl().build() + qD2;
                                addDecreasedScoreResults(worsenedResults, questionId, questionDescription, improvedResults, worsenedResults, noChangeResults);
                            }
							else if (Math.abs(valueChange) == 0){
                                String qD2 = HtmlReportBuilder.create(String.format(" - %1s", value)).nl().build();
                                String questionDescription = qD1 + qD2;
                                addNoChangeScoreResults(noChangeResults, questionId, questionDescription, improvedResults, worsenedResults,noChangeResults);
                            }
							
						}

					}

				}
			}
		}


        resultsSummary += addImprovedWorsenedChangeResults(HtmlReportBuilder.create(reportSettings.improvedChangeHeaderText).underline().bold().nl().build(), improvedResults);
        resultsSummary += addImprovedWorsenedChangeResults(HtmlReportBuilder.create(reportSettings.worsenedChangeHeaderText).underline().bold().nl().build(), worsenedResults);
        resultsSummary += addNoChangeResults(HtmlReportBuilder.create(reportSettings.noChangeHeaderText).underline().bold().nl().build(), noChangeResults);

		resultsSummary += reportSettings.endOfAllResultsText;

        //logger.debug(resultsSummary);
		return new AssessmentProperty(AssessmentCode.REPORT_SUMMARY, resultsSummary);

	}

    //Question 8 and 13 are better if score increases
    private void addIncreasedScoreResults(ArrayList<String> increasedResults, String questionId, String questionDescription, ArrayList<String> improvedResults, ArrayList<String> worsenedResults, ArrayList<String> noChangeResults){
        if ("Q8".equals(questionId) || "Q13".equals(questionId)){
            improvedResults.add(questionDescription);
        }
        else{
            worsenedResults.add(questionDescription);
        }

    }
    //Question 8 and 13 are worse if score decreases
    private void addDecreasedScoreResults(ArrayList<String> increasedResults, String questionId, String questionDescription, ArrayList<String> improvedResults, ArrayList<String> worsenedResults, ArrayList<String> noChangeResults){
        if ("Q8".equals(questionId) || "Q13".equals(questionId)){
            worsenedResults.add(questionDescription);
        }
        else{
            improvedResults.add(questionDescription);
        }

    }
    private void addNoChangeScoreResults(ArrayList<String> increasedResults, String questionId, String questionDescription, ArrayList<String> improvedResults, ArrayList<String> worsenedResults, ArrayList<String> noChangeResults){
        noChangeResults.add(questionDescription);
    }

    private String getQuestionDescription(String questionId){
        return PainConsolidatedReportHelper.getMonthlyPainAssessmentQuestionDescription(questionId);
    }

	private String getResponseSelectedValue(AssessmentResponse response) {
		return response.getSelectedChoices().get(0).getProperties().getValueByPropertyName(AssessmentCode.RESULT_PROPERTY_SELECTED_VALUE);
	}

    private String addImprovedWorsenedChangeResults(String changesHeader, ArrayList<String> changeResults){

        return addResultChanges(changesHeader, changeResults, true);
    }

    private String addNoChangeResults(String changesHeader, ArrayList<String> changeResults){

        return addResultChanges(changesHeader, changeResults, false);
    }

	private String addResultChanges(String changesHeader, ArrayList<String> changeResults, boolean newLineAfterEachResult) {
        String result = "";
		
		if (changeResults.size() > 0){
			result = changesHeader;
			for (String item : changeResults) {
                if (newLineAfterEachResult){
                    result += HtmlReportBuilder.create(item).nl().build();
                }
                else{
                    result += HtmlReportBuilder.create(item).build();
                }

			}
		}
        result = HtmlReportBuilder.create(result).nl().build();
		return result;
	}
}