package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentEligibility;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.utility.DateHelper;

public class EligibilityCalculator {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(EligibilityCalculator.class);
	private AssessmentEligibility eligibility = new AssessmentEligibility();

	public EligibilityCalculator() {
	}

	public EligibilityCalculator(String assessmentId, PatientIdentifier patientIdentifier) {
		this.eligibility = new AssessmentEligibility(assessmentId, patientIdentifier);
	}

	public AssessmentEligibility determineEligibility(int waitingPeriod, AssessmentResult mostRecentlyCompletedResult) {
		boolean assessmentHasBeenTaken = mostRecentlyCompletedResult != null;
		if (!assessmentHasBeenTaken) {
			setEligible();
		} else {
			Date dateTaken = mostRecentlyCompletedResult.getDateTaken();
			Date dateTakenBeginOfDay = DateHelper.beginOfDate(dateTaken);
			int daysSincePrevious = DateHelper.calculateDeltaInDays(dateTakenBeginOfDay, new Date());
			boolean needToWait = waitingPeriod > 0 && daysSincePrevious < waitingPeriod;
			logger.debug(String.format("Waiting period is [%s], last assessment was taken on [%s], it has been [%s] days since previous assessment, still need to wait is %s",waitingPeriod,dateTaken, daysSincePrevious, needToWait));
			if (needToWait) {
				setNotEligible(daysSincePrevious, waitingPeriod);
			} else {
				setEligible();
			}
		}
		return eligibility;
	}

	private void setEligible() {
		eligibility.setEligibility(true);
		eligibility.setDaysUntilEligible(0);
		logger.debug("setting eligible");
	}

	private void setNotEligible(int daysSincePrevious, int waitingPeriod) {
		int daysUntilEligible = waitingPeriod - daysSincePrevious;

		eligibility.setEligibility(false);
		eligibility.setDaysUntilEligible(daysUntilEligible);
		eligibility.setMessage(generateNotEligibleMessage(waitingPeriod, daysUntilEligible));
		logger.debug("setting not eligible");
	}

	private String generateNotEligibleMessage(int waitingPeriod, int daysUntilEligible) {
		String retakeLengthText = translateNumberOfDaysToText(waitingPeriod);
		String daysLeftText = translateNumberOfDaysToText(daysUntilEligible);
		String messageToFormat = "It has been less than %s since you last took the self-assessment. The questions are designed to assess your symptoms since the last time you answered, so it is not helpful to take it more than once in %s. ";
		String messageToSet = String.format(messageToFormat, retakeLengthText, retakeLengthText, daysLeftText);
		if(daysUntilEligible>0) {
			messageToSet += String.format("Please plan to retake the assessment in %s.",daysLeftText);
		}
		return messageToSet;
	}

	private String translateNumberOfDaysToText(int i) {
		if (i == 7) {
			return "one week";
		} else if (i == 1) {
			return "one day";
		}
		return i + " days";
	}
}
