package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import com.agilex.healthcare.mobilehealthplatform.datalayer.assessment.AssessmentDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports.PainConsolidatedReportGenerator;
import com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults.reports.PainConsolidatedReportGeneratorParameter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataEditRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataResponseReader;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DomainServiceRegistry;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.PatientIdentifierHelper;
import com.agilex.healthcare.utility.DateHelper;

public class AssessmentResultsDataService {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(AssessmentResultsDataService.class);
	private AssessmentDataService assessmentDataService = new AssessmentDataService();

	// todo: review how to determine anonymous patient identifier
	private final PatientIdentifier anonymousPatientIdentifier = AssessmentCode.ANONYMOUS_RESULT_PATIENT_ID;

	private Router router;

	public AssessmentResultsDataService() {
		this.router = new Router();
	}

	public AssessmentResultsDataService(Router router) {
		this.router = router;
	}

	public AssessmentEligibility fetchAssessmentEligibility(String assessmentId, PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
		AssessmentResult mostRecentlyCompletedResult = fetchMostRecentlyCompletedResult(assessmentId, patientIdentifier, scopeFilter);

		Assessment completedAssessment = assessmentDataService.fetchAssessment(assessmentId);
		int waitingPeriod = completedAssessment.getWaitingPeriod();

		EligibilityCalculator calculator = new EligibilityCalculator(assessmentId, patientIdentifier);
		return calculator.determineEligibility(waitingPeriod, mostRecentlyCompletedResult);
	}

	public AssessmentResult fetchAssessmentResult(PatientIdentifier patientIdentifier, String resultId) {
		AssessmentResult retrievedAssessmentResult = getAssessmentResultDataLayer().fetchAssessmentResult(patientIdentifier, resultId);
		PatientIdentifierHelper.updatePatientIdentifier(retrievedAssessmentResult, patientIdentifier);
		return retrievedAssessmentResult;
	}

	public AssessmentResults fetchResultsByAssessment(PatientIdentifier patientIdentifier, String assessmentId, DateFilter filter, ScopeFilter scopeFilter) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.assessmentResult).forPatientIdentifier(patientIdentifier).forDateFilter(filter).forScopeFilter(scopeFilter).setValue("assessment-id", assessmentId).build();

		ResponseMessage responseMessage = router.execute(request);
		AssessmentResults assessmentResults = PatientDataResponseReader.<AssessmentResults, AssessmentResult> fromResponse(responseMessage).getDataListNoNull(AssessmentResults.class);

		if (assessmentId.equals("ptsd")) {
			request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.pclAssessmentResult).forPatientIdentifier(patientIdentifier).forDateFilter(filter).forScopeFilter(scopeFilter).setValue("assessment-id", assessmentId).build();
			responseMessage = router.execute(request);
			AssessmentResults historicPclScores = PatientDataResponseReader.<AssessmentResults, AssessmentResult> fromResponse(responseMessage).getDataListNoNull(AssessmentResults.class);

			assessmentResults.addAll(historicPclScores);
		}

		PatientIdentifierHelper.updatePatientIdentifier(assessmentResults, patientIdentifier);
		return assessmentResults;
	}

	public AssessmentResults fetchAssessmentDraftsByPatient(PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.assessmentResult).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).setChildMessageType("{domain}.fetchDrafts").build();

		ResponseMessage responseMessage = router.execute(request);
		AssessmentResults assessmentResults = PatientDataResponseReader.<AssessmentResults, AssessmentResult> fromResponse(responseMessage).getDataListNoNull(AssessmentResults.class);

		return assessmentResults;
	}

	public boolean checkPatientHasAppointmentWithinTwoWeeks(PatientIdentifier patientIdentifier) {
		DateFilter dateFilter = DateFilterFactory.createFilterFromDate(DateHelper.getToday(), DateHelper.plusDays(DateHelper.getToday(), 14));
		RequestMessage requestMessage = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.mentalHealthAppointment).forPatientIdentifier(patientIdentifier).forDateFilter(dateFilter).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage response = router.execute(requestMessage);
		
		Appointments appointments = PatientDataResponseReader.<Appointments, Appointment> fromResponse(response).getDataListNoNull(Appointments.class);
		return (appointments.size() > 0);
	}

	public AssessmentResult saveResult(AssessmentResult assessmentResult, ScopeFilter scopeFilter) {
		logger.debug("request to save assessment result");
		fillAssessmentData(assessmentResult);

		checkForAnonymousAssessment(assessmentResult);

		int calculatedScore = ResultScorerFactory.getScorer(assessmentResult).calculateScore(assessmentResult);
		assessmentResult.setScore(calculatedScore);
		logger.debug("scored assessment result: " + assessmentResult.getScore());

		AssessmentResult previousResult = fetchMostRecentlyCompletedResult(assessmentResult.getAssessmentId(), assessmentResult.getPatientIdentifier(), scopeFilter);

		AssessmentResultReport generatedReport = ReportGeneratorFactory.getReportGenerator(assessmentResult).generateReport(assessmentResult, previousResult);
		assessmentResult.setAssessmentResultReport(generatedReport);

		logger.debug("about to save assessment result");

		addSurrogate(assessmentResult);

		RequestMessage message = PatientDataEditRequestBuilder.forCreate().forDomain(Domain.assessmentResult).forData(assessmentResult).forScope(scopeFilter).build();
		ResponseMessage response = router.execute(message);
		AssessmentResult savedAssessmentResult = PatientDataResponseReader.<AssessmentResults, AssessmentResult> fromResponse(response).getDataItem();

		logger.debug("saved assessment result");

		if (!savedAssessmentResult.getPatientIdentifier().equals(anonymousPatientIdentifier)) {
			savedAssessmentResult.setPatientIdentifier(assessmentResult.getPatientIdentifier());
		}

		return savedAssessmentResult;
	}

	private void addSurrogate(AssessmentResult data) {
		MhpUserFactory.addSurrogateToPatientData(data);
	}

	public AssessmentResultReport getPainConsolidatedReport(PainConsolidatedReportGeneratorParameter reportGeneratorParameter) {
		PainConsolidatedReportGenerator generator = new PainConsolidatedReportGenerator(reportGeneratorParameter);
		return generator.generateReport();
	}

	private void fillAssessmentData(AssessmentResult assessmentResult) {
		Assessment correspondingAssessment = assessmentDataService.fetchAssessment(assessmentResult.getAssessmentId());
		if (correspondingAssessment != null) {
			assessmentResult.setVersion(correspondingAssessment.getVersion());
			assessmentResult.setAuthenticationStrategy(correspondingAssessment.getAuthenticationStrategy());
			assessmentResult.setNotes(correspondingAssessment.getNotes());
			assessmentResult.setScoringAlgorithm(correspondingAssessment.getScoringAlgorithm());
			assessmentResult.setWaitingPeriod(correspondingAssessment.getWaitingPeriod());
		}
	}

	private void checkForAnonymousAssessment(AssessmentResult assessmentResult) {
		logger.debug("checking for anonymous assessment");

		boolean assessmentIsAnonymous = assessmentResult.getAuthenticationStrategy().equals(AssessmentCode.AUTHENTICATION_STRATEGY_ANONYMOUS);
		if (assessmentIsAnonymous) {
			assessmentResult.setPatientIdentifier(anonymousPatientIdentifier);
			logger.debug("set user for assessment result to anonymous");
		}
	}

	private AssessmentResultDataLayer getAssessmentResultDataLayer() {
		return getAssessmentResultDataLayer(ScopeFilter.getInstanceForLongitudinalScope());
	}

	private AssessmentResultDataLayer getAssessmentResultDataLayer(ScopeFilter scopeFilter) {
		DomainServiceRegistry serviceRegistry = new DomainServiceRegistry();
		return serviceRegistry.getDataLayerByScopeFilter(scopeFilter, "assessmentresult");
	}

	private AssessmentResult fetchMostRecentlyCompletedResult(String assessmentId, PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.assessmentResult).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).setChildMessageType("{domain}.fetchMostRecentlyCompletedResult").setValue("assessment-id", assessmentId).build();

		ResponseMessage responseMessage = router.execute(request);
		AssessmentResults assessmentResults = PatientDataResponseReader.<AssessmentResults, AssessmentResult> fromResponse(responseMessage).getDataListNoNull(AssessmentResults.class);

		AssessmentResult mostRecentlyCompletedResult = null;
		if (assessmentResults != null && assessmentResults.size() >= 1) {
			mostRecentlyCompletedResult = assessmentResults.get(0);
		}
		return mostRecentlyCompletedResult;

	}

}
