package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import java.util.Date;
import java.util.List;

import javax.ws.rs.WebApplicationException;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataEditRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataResponseReader;
import com.agilex.healthcare.mobilehealthplatform.datalayer.facility.FacilityDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestDetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessages;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequests;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Facilities;
import com.agilex.healthcare.mobilehealthplatform.domain.Facility;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestType;
import com.agilex.healthcare.mobilehealthplatform.enumeration.RunningMode;
import com.agilex.healthcare.mobilehealthplatform.restservice.exception.InvalidAppointmentRequestException;
import com.agilex.healthcare.mobilehealthplatform.restservice.exception.InvalidAppointmentRequestMessageException;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.validator.appointment.AppointmentRequestValidator;
import com.agilex.healthcare.mobilehealthplatform.validator.appointment.AppointmentRequestValidatorFactory;
import com.agilex.healthcare.mobilehealthplatform.validator.appointmentrequestmessage.AppointmentRequestMessageValidator;
import com.agilex.healthcare.mobilehealthplatform.validator.appointmentrequestmessage.AppointmentRequestMessageValidatorFactory;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;

public class AppointmentRequestDataService {
	private static final int EarliestTimeVeteranCanRequestAppointmentInHours = 96;
	private static final int FarthestTimeVeteranCanRequestAppointmentInDays = 120;
    private static final int messageLimitPerSender = 2;

	private static final String LAST_UPDATED_DATE = "lastUpdatedDate";
	private static final String PRIMARY_CARE_MAX_LIMIT_ERROR_MSG = "**Number of Request Exceeded**<br/>You can submit one primary care appointment request at a time. "
			+ "Another request may be submitted after the current primary care request is processed by the VA, or you may return to the previous screen, cancel the pending request, and submit a new one.";
	private static final String MENTAL_HEALTH_MAX_LIMIT_ERROR_MSG = "**Number of Request Exceeded**<br/>You can only have two pending mental health appointment requests at a time. "
			+ "Another request may be submitted after one or more existing mental health appointment requests are processed by the VA, or you may return to the previous screen, cancel a pending request, and submit a new one.";
	private static final String FACILITY_NOT_FOUND_ERROR_MSG = "The requested facility was not found.  Please choose the desired facility using the autocomplete feature.";
	private static final String STATUS_NOT_FOUND_ERROR_MSG = "The appointment request is missing a valid status.";
	private static final String INVALID_STATUS_ERROR_MSG = "The requested appointment request status is invalid.";
    private static final String APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_FOR_CANCELLED_APPOINTMENT = "Appointment request messages are not allowed for cancelled appointment requests.";
    private static final String APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_AFTER_APPOINTMENT_DATE_PASSED = "Appointment request messages are not allowed after appointment date has passed.";
    private static final String APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_UNTIL_REPLY = "New appointment request message not allowed until reply received.";
    private static final String APPOINTMENT_REQUEST_MSG_LIMIT_REACHED = "Maximum allowed number of messages for this appointment request reached.";

	private Router router;

	public AppointmentRequestDataService() {
		this.router = new Router();
	}

	public AppointmentRequestDataService(Router router) {
		this.router = router;
	}

	public AppointmentRequests getPatientAppointmentRequests(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.appointmentRequest).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).forDateFilter(dateFilter).excludeDetails().build();
		ResponseMessage responseMessage = router.execute(request);

		AppointmentRequests appointmentRequests = readAppointmentRequestsFromRouterResponseAndSortDescendingByLastUpdatedDate(responseMessage);
		return appointmentRequests;
	}

	public Facilities fetchMostRecentFacilities(PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
		AppointmentRequests patientAppointmentRequests = getPatientAppointmentRequests(patientIdentifier, DateFilterFactory.createEmptyFilter(), scopeFilter);
		patientAppointmentRequests.sortDescending("createdDate");

		Facilities facilities = new Facilities();

		for (AppointmentRequest appointmentRequest : patientAppointmentRequests) {
			Facility facility = appointmentRequest.getFacility();

			if (facilities.size() < 3) {
				if (!facilities.contains(facility)) {
					facilities.add(facility);
				}
			} else {
				break;
			}
		}

		return facilities;
	}

	public AppointmentRequests getAppointmentRequests(AppointmentRequestFilter filter) {
		RequestMessage request = AppointmentRequestSearchRequestBuilder.forRetrieveList().forAppointmentRequestFilter(filter).forDomain(Domain.appointmentRequest).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);

		AppointmentRequests appointmentRequests = readAppointmentRequestsFromRouterResponseAndSortAscendingByLastUpdatedDate(responseMessage);
		AppointmentRequests filteredAppointmentRequests = excludeAppointmentRequestsCancelledByVeteranBeforeBeingBooked(appointmentRequests);

		return filteredAppointmentRequests;
	}

    public AppointmentRequests getAppointmentRequestsNoSort(AppointmentRequestFilter filter) {
        RequestMessage request = AppointmentRequestSearchRequestBuilder.forRetrieveList().forAppointmentRequestFilter(filter).forDomain(Domain.appointmentRequest).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).build();
        ResponseMessage responseMessage = router.execute(request);
        AppointmentRequests appointmentRequests = PatientDataResponseReader.<AppointmentRequests, AppointmentRequest> fromResponse(responseMessage).getDataListNoNull(AppointmentRequests.class);
        AppointmentRequests filteredAppointmentRequests = excludeAppointmentRequestsCancelledByVeteranBeforeBeingBooked(appointmentRequests);

        return filteredAppointmentRequests;
    }


	private AppointmentRequests excludeAppointmentRequestsCancelledByVeteranBeforeBeingBooked(AppointmentRequests appointmentRequests) {
		AppointmentRequests filteredAppointmentRequests = new AppointmentRequests();

		for (AppointmentRequest appointmentRequest : appointmentRequests) {
			if (isCancelledAppointmentRequest(appointmentRequest)) {
				if (doesAppointmentRequestContainDetailCode(appointmentRequest)) {
					filteredAppointmentRequests.add(appointmentRequest);
				}
			} else {
				filteredAppointmentRequests.add(appointmentRequest);
			}
		}

		return filteredAppointmentRequests;
	}

	private boolean doesAppointmentRequestContainDetailCode(AppointmentRequest appointmentRequest) {
		List<AppointmentRequestDetailCode> arDetailCodes = appointmentRequest.getAppointmentRequestDetailCode();
		if (NullChecker.isNotNullish(arDetailCodes)) {
			return true;
		}
		return false;
	}

	private boolean isCancelledAppointmentRequest(AppointmentRequest appointmentRequest) {
		if (AppointmentRequestStatus.CANCELLED.getName().equalsIgnoreCase(appointmentRequest.getStatus())) {
			return true;
		}
		return false;
	}

	private AppointmentRequests readAppointmentRequestsFromRouterResponseAndSortDescendingByLastUpdatedDate(ResponseMessage responseMessage) {
		AppointmentRequests appointmentRequests = PatientDataResponseReader.<AppointmentRequests, AppointmentRequest> fromResponse(responseMessage).getDataListNoNull(AppointmentRequests.class);
		//appointmentRequests.sortDescending(LAST_UPDATED_DATE);
		return appointmentRequests;
	}

	private AppointmentRequests readAppointmentRequestsFromRouterResponseAndSortAscendingByLastUpdatedDate(ResponseMessage responseMessage) {
		AppointmentRequests appointmentRequests = PatientDataResponseReader.<AppointmentRequests, AppointmentRequest> fromResponse(responseMessage).getDataListNoNull(AppointmentRequests.class);
		appointmentRequests.sortAscending(LAST_UPDATED_DATE);
		return appointmentRequests;
	}

	public AppointmentRequest getPatientAppointmentRequest(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveSingleById().forDomain(Domain.appointmentRequest).forPatientIdentifier(patientIdentifier).forDataIdentifier(dataIdentifier).build();
		ResponseMessage responseMessage = router.execute(request);
		AppointmentRequest appointmentRequest = PatientDataResponseReader.<AppointmentRequests, AppointmentRequest> fromResponse(responseMessage).getDataItem();

		return appointmentRequest;
	}

	public AppointmentRequestMessages fetchAppointmentRequestMessages(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier) {
		RequestMessage request = AppointmentRequestSearchRequestBuilder.forRetrieveMessageList().forDomain(Domain.appointmentRequest).forPatientIdentifier(patientIdentifier).forDataIdentifier(dataIdentifier).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);

		AppointmentRequestMessages messages = readAppointmentRequestMessagesFromRouterResponseAndSortDescendingMessageDateTime(responseMessage);
		return messages;
	}

	private AppointmentRequestMessages readAppointmentRequestMessagesFromRouterResponseAndSortDescendingMessageDateTime(ResponseMessage responseMessage) {
		AppointmentRequestMessages messages = PatientDataResponseReader.<AppointmentRequestMessages, AppointmentRequestMessage> fromResponse(responseMessage).getDataListNoNull(AppointmentRequestMessages.class);
		messages.sortDescending("messageDateTime");
		return messages;
	}

	public AppointmentRequest saveAppointmentRequest(List<String> pilotSites, AppointmentRequest appointmentRequest, String haMode, boolean fullValidation, ScopeFilter scopeFilter) {
		Patient patient = null;
		if (haMode.equalsIgnoreCase(RunningMode.VETERAN.getName())) {
			patient = MhpUserFactory.createFromSecurityContext().getPatient();
		} else {
			patient = appointmentRequest.getPatient();
		}

		if (NullChecker.isNullish(patient.getPatientIdentifier().getUniqueId())) {
			patient.setPatientIdentifier(appointmentRequest.getPatientIdentifier());
		}
        
		patient.setTextMessagingAllowed(appointmentRequest.isTextMessagingAllowed());
		patient.setTextMessagingPhoneNumber(appointmentRequest.getTextMessagingPhoneNumber());

		appointmentRequest.setPatient(patient);
		if (appointmentRequest.getParentRequest() != null) {
			appointmentRequest.getParentRequest().setPatient(patient);
		}

		validateAppointmentRequest(pilotSites, appointmentRequest, fullValidation);

        addSurrogate(appointmentRequest);
		RequestMessage request = PatientDataEditRequestBuilder.forUpdate().forDomain(Domain.appointmentRequest).forScope(scopeFilter).forData(appointmentRequest).build();
		ResponseMessage responseMessage = router.execute(request);

		AppointmentRequest result = PatientDataResponseReader.<AppointmentRequests, AppointmentRequest> fromResponse(responseMessage).getDataItem();

		return result;
	}

	public AppointmentRequestMessage saveAppointmentRequestMessage(AppointmentRequestMessage appointmentRequestMessage) {

        AppointmentRequest associatedAppointmentRequest = getPatientAppointmentRequest(appointmentRequestMessage.getPatientIdentifier(), appointmentRequestMessage.getDataIdentifier());

        validateAppointmentRequestMessage(appointmentRequestMessage, associatedAppointmentRequest);

        addSurrogate(appointmentRequestMessage);
        RequestMessage request = AppointmentRequestMessageEditRequestBuilder.forCreateAppointmentRequestMessage().forDomain(Domain.appointmentRequest).forScope(ScopeFilter.getInstanceForLongitudinalScope()).forData(appointmentRequestMessage).build();
		ResponseMessage responseMessage = router.execute(request);

		AppointmentRequestMessage result = PatientDataResponseReader.<AppointmentRequestMessages, AppointmentRequestMessage> fromResponse(responseMessage).getDataItem();

		return result;
	}

    private void validateAppointmentRequestMessage(AppointmentRequestMessage appointmentRequestMessage, AppointmentRequest associatedAppointmentRequest) {

        AppointmentRequestMessageValidator validator = AppointmentRequestMessageValidatorFactory.createAppointmentRequestMessageValidator();

        if (validator != null) {
            ValidationResult<AppointmentRequestMessage> validationResult = validator.validate(appointmentRequestMessage);
            if (!validationResult.isValid()) {
                throw new InvalidAppointmentRequestMessageException(validationResult.getErrors());
            }
        }

        throwAnErrorIfAppointmentRequestIsCancelled(associatedAppointmentRequest, appointmentRequestMessage);
        throwAnErrorIfAppointmentRequestDateHasPassed(associatedAppointmentRequest, appointmentRequestMessage);
        throwAnErrorIfNewMessageHasNotReceivedReply(associatedAppointmentRequest, appointmentRequestMessage);
        throwAnErrorIfMessageLimitReached(associatedAppointmentRequest, appointmentRequestMessage);
    }

    public AppointmentRequest updateAppointmentMessageFlag(String appointmentRequestId, PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier){
		RequestMessage request = UpdateAppointmentRequestMessageFlagRequestBuilder.forUpdateMessageFlag().forDomain(Domain.appointmentRequest).forPatientIdentifier(patientIdentifier).forDataIdentifier(dataIdentifier).forAppointmentRequestId(appointmentRequestId).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).build();
		ResponseMessage responseMessage = router.execute(request);
		AppointmentRequest appointmentRequest = PatientDataResponseReader.<AppointmentRequests, AppointmentRequest> fromResponse(responseMessage).getDataItem();
		return appointmentRequest;
	}
	
	public AppointmentRequest deleteAppointmentRequest(AppointmentRequest appointmentRequest, ScopeFilter scopeFilter) {
        addSurrogate(appointmentRequest);
		RequestMessage request = PatientDataEditRequestBuilder.forDelete().forDomain(Domain.appointmentRequest).forScope(scopeFilter).forData(appointmentRequest).build();
		ResponseMessage responseMessage = router.execute(request);
		return PatientDataResponseReader.<AppointmentRequests, AppointmentRequest> fromResponse(responseMessage).getDataItem();
	}

	private void validateAppointmentRequest(List<String> pilotSites, AppointmentRequest appointmentRequest, boolean fullValidation) {
		if (fullValidation) {
			if (AppointmentRequestStatus.SUBMITTED.getName().equalsIgnoreCase(appointmentRequest.getStatus()) && !appointmentRequest.isSecondRequest()) {
				validateMaximumNumberOfAppointmentRequests(appointmentRequest);
			}
		}

		AppointmentRequestValidator validator = AppointmentRequestValidatorFactory.createAppointmentRequestValidator(EarliestTimeVeteranCanRequestAppointmentInHours, FarthestTimeVeteranCanRequestAppointmentInDays);

		if (validator != null) {
			ValidationResult<AppointmentRequest> validationResult = validator.validate(appointmentRequest, fullValidation);
			if (!validationResult.isValid()) {
				throw new InvalidAppointmentRequestException(validationResult.getErrors());
			}
		}

		validateFacilityCode(pilotSites, appointmentRequest);
		validateStatus(appointmentRequest);
	}

	private void validateStatus(AppointmentRequest appointmentRequest) {
		String status = appointmentRequest.getStatus();

		if (NullChecker.isNotNullish(status)) {
			boolean found = false;
			AppointmentRequestStatus[] statuses = AppointmentRequestStatus.values();
			for (AppointmentRequestStatus statusEnum : statuses) {
				if (status.equals(statusEnum.getName())) {
					found = true;
					break;
				}
			}

			if (!found) {
				ValidationResult<AppointmentRequest> validationResult = createOpenAppointmentRequestValidationError(appointmentRequest, INVALID_STATUS_ERROR_MSG);
				throw new InvalidAppointmentRequestException(validationResult.getErrors());
			}
		} else {
			ValidationResult<AppointmentRequest> validationResult = createOpenAppointmentRequestValidationError(appointmentRequest, STATUS_NOT_FOUND_ERROR_MSG);
			throw new InvalidAppointmentRequestException(validationResult.getErrors());
		}
	}

	private void validateFacilityCode(List<String> pilotSites, AppointmentRequest appointmentRequest) {
		boolean notFound = false;
		try {
			String facilityCode = appointmentRequest.getFacility().getFacilityCode();
			FacilityDataService facilityDataService = new FacilityDataService();
			Facility facility = facilityDataService.fetchFacilityByCode(pilotSites, facilityCode);

			if (facility == null) {
				notFound = true;
			}
		} catch (WebApplicationException e) {
			notFound = true;
		}

		if (notFound) {
			ValidationResult<AppointmentRequest> validationResult = createValidationError(appointmentRequest, "facilityName", FACILITY_NOT_FOUND_ERROR_MSG);
			throw new InvalidAppointmentRequestException(validationResult.getErrors());
		}
	}

	private void validateMaximumNumberOfAppointmentRequests(AppointmentRequest appointmentRequest) {
		if (AppointmentRequestType.PRIMARY_CARE.getName().equals(appointmentRequest.getAppointmentType())) {
			throwAnErrorIfPrimaryCareAppointmentRequestsExist(appointmentRequest);
		} else if (AppointmentRequestType.MENTAL_HEALTH.getName().equals(appointmentRequest.getAppointmentType())) {
			throwAnErrorIfMentalHealthAppointmentRequestsExist(appointmentRequest);
		}
	}

	private void throwAnErrorIfMentalHealthAppointmentRequestsExist(AppointmentRequest appointmentRequest) {
		if (areThereMoreThanOneSubmittedMentalHealthAppointmentRequests(appointmentRequest)) {
			ValidationResult<AppointmentRequest> validationResult = createOpenAppointmentRequestValidationError(appointmentRequest, MENTAL_HEALTH_MAX_LIMIT_ERROR_MSG);
			throw new InvalidAppointmentRequestException(validationResult.getErrors());
		}
	}

	private boolean areThereMoreThanOneSubmittedMentalHealthAppointmentRequests(AppointmentRequest submittedAppointmentRequest) {
		DateFilter defaultDateFilter = createDateFilterWithDefaultDateRange();

		AppointmentRequests patientAppointmentRequests = getPatientAppointmentRequests(submittedAppointmentRequest.getPatientIdentifier(), defaultDateFilter, ScopeFilter.getInstanceForLongitudinalScope());

		int recordCount = 0;
		for (AppointmentRequest appointmentRequest : patientAppointmentRequests) {
			if (appointmentRequest.getAppointmentType().equals(AppointmentRequestType.MENTAL_HEALTH.getName()) && appointmentRequest.getStatus().equals(AppointmentRequestStatus.SUBMITTED.getName())) {
				if (++recordCount > 1) {
					return true;
				}
			}
		}

		return false;
	}

	private void throwAnErrorIfPrimaryCareAppointmentRequestsExist(AppointmentRequest appointmentRequest) {
		if (areThereAnySubmittedPrimaryCareAppointmentRequests(appointmentRequest)) {
			ValidationResult<AppointmentRequest> validationResult = createOpenAppointmentRequestValidationError(appointmentRequest, PRIMARY_CARE_MAX_LIMIT_ERROR_MSG);
			throw new InvalidAppointmentRequestException(validationResult.getErrors());
		}
	}

	private ValidationResult<AppointmentRequest> createOpenAppointmentRequestValidationError(AppointmentRequest appointmentRequest, String errorMessage) {
		return createValidationError(appointmentRequest, "error", errorMessage);
	}

	private ValidationResult<AppointmentRequest> createValidationError(AppointmentRequest appointmentRequest, String fieldName, String errorMessage) {
		ValidationResult<AppointmentRequest> validationResult = new ValidationResult<AppointmentRequest>();
		validationResult.setRequestObject(appointmentRequest);

		ValidationError error = new ValidationError();
		error.setErrorMessage(errorMessage);
		error.setFieldName(fieldName);

		validationResult.addIfNotNull(error);
		return validationResult;
	}

    private void throwAnErrorIfAppointmentRequestDateHasPassed(AppointmentRequest appointmentRequest, AppointmentRequestMessage appointmentRequestMessage) {
        if (!hasAppointmentDatePassed(appointmentRequest.getAppointmentDate())){
            ValidationResult<AppointmentRequestMessage> validationResult = createOpenAppointmentRequestMessageValidationError(appointmentRequestMessage, APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_AFTER_APPOINTMENT_DATE_PASSED);
            throw new InvalidAppointmentRequestMessageException(validationResult.getErrors());
        }
    }
    private boolean hasAppointmentDatePassed(String appointmentRequestDate){

        if(NullChecker.isNotNullish(appointmentRequestDate)){
            Date today = DateHelper.beginOfDate(DateHelper.getToday());
            Date appointmentDate = DateHelper.beginOfDate(DateHelper.parseDate(appointmentRequestDate));
            if(today.after(appointmentDate))
                return false;
        }

        return true;
    }

    private void throwAnErrorIfAppointmentRequestIsCancelled(AppointmentRequest appointmentRequest, AppointmentRequestMessage appointmentRequestMessage) {
        if (appointmentRequest.getStatus().equals("Cancelled")){
            ValidationResult<AppointmentRequestMessage> validationResult = createOpenAppointmentRequestMessageValidationError(appointmentRequestMessage, APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_FOR_CANCELLED_APPOINTMENT);
            throw new InvalidAppointmentRequestMessageException(validationResult.getErrors());
        }
    }

    private ValidationResult<AppointmentRequestMessage> createOpenAppointmentRequestMessageValidationError(AppointmentRequestMessage appointmentRequestMessage, String errorMessage) {
        ValidationResult<AppointmentRequestMessage> validationResult = new ValidationResult<AppointmentRequestMessage>();
        validationResult.setRequestObject(appointmentRequestMessage);

        ValidationError error = new ValidationError();
        error.setErrorMessage(errorMessage);
        error.setFieldName("error");

        validationResult.addIfNotNull(error);
        return validationResult;
    }


    private void throwAnErrorIfNewMessageHasNotReceivedReply(AppointmentRequest appointmentRequest, AppointmentRequestMessage appointmentRequestMessage){

       AppointmentRequestMessages currentMessagesForAppointment =  fetchAppointmentRequestMessages(appointmentRequest.getPatientIdentifier(), appointmentRequest.getDataIdentifier());

       if(!currentMessagesForAppointment.isEmpty()){
           AppointmentRequestMessage latestMessage = currentMessagesForAppointment.get(0);
           if(latestMessage.getSenderId().equals(appointmentRequestMessage.getSenderId())){
               ValidationResult<AppointmentRequestMessage> validationResult = createOpenAppointmentRequestMessageValidationError(appointmentRequestMessage, APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_UNTIL_REPLY);
               throw new InvalidAppointmentRequestMessageException(validationResult.getErrors());
           }
       }


    }


    private void throwAnErrorIfMessageLimitReached(AppointmentRequest appointmentRequest, AppointmentRequestMessage appointmentRequestMessage){

        AppointmentRequestMessages currentMessagesForAppointment =  fetchAppointmentRequestMessages(appointmentRequest.getPatientIdentifier(), appointmentRequest.getDataIdentifier());
        int numberOfMessagesAlreadySentBySender = 0;
        for(AppointmentRequestMessage aptReqMsg : currentMessagesForAppointment){
            if(appointmentRequestMessage.getSenderId().equals(aptReqMsg.getSenderId())){
                numberOfMessagesAlreadySentBySender ++;
            }
        }

        if(numberOfMessagesAlreadySentBySender >= messageLimitPerSender){
            ValidationResult<AppointmentRequestMessage> validationResult = createOpenAppointmentRequestMessageValidationError(appointmentRequestMessage, APPOINTMENT_REQUEST_MSG_LIMIT_REACHED);
            throw new InvalidAppointmentRequestMessageException(validationResult.getErrors());
        }

    }


    private boolean areThereAnySubmittedPrimaryCareAppointmentRequests(AppointmentRequest submittedAppointmentRequest) {
		DateFilter defaultDateFilter = createDateFilterWithDefaultDateRange();

		AppointmentRequests patientAppointmentRequests = getPatientAppointmentRequests(submittedAppointmentRequest.getPatientIdentifier(), defaultDateFilter, ScopeFilter.getInstanceForLongitudinalScope());

		for (AppointmentRequest appointmentRequest : patientAppointmentRequests) {
			if (appointmentRequest.getAppointmentType().equals(AppointmentRequestType.PRIMARY_CARE.getName()) && appointmentRequest.getStatus().equals(AppointmentRequestStatus.SUBMITTED.getName())) {
				return true;
			}
		}

		return false;
	}

	private DateFilter createDateFilterWithDefaultDateRange() {
		Date endDate = DateHelper.getToday();
		Date startDate = DateHelper.beginOfDate(DateHelper.minusDays(endDate, FarthestTimeVeteranCanRequestAppointmentInDays));
		return DateFilterFactory.createFilterFromDate(startDate, endDate);
	}
	
	public AppointmentRequest updateProviderSeeAppointmentRequestFlag(String appointmentRequestId, PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier){
		RequestMessage request = UpdateProviderSeenAppointmentRequestFlagRequestBuilder.forUpdateProviderSeenAppointmentRequestFlag().forDomain(Domain.appointmentRequest).forPatientIdentifier(patientIdentifier).forDataIdentifier(dataIdentifier).forScopeFilter(ScopeFilter.getInstanceForLongitudinalScope()).forAppointmentRequestId(appointmentRequestId).build();
		ResponseMessage responseMessage = router.execute(request);
		return PatientDataResponseReader.<AppointmentRequests, AppointmentRequest>fromResponse(responseMessage).getDataItem();
	}
	
	private void addSurrogate(PatientData patientData) {
        MhpUserFactory.addSurrogateToPatientData(patientData);
	}
}
