package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import java.util.Collections;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataResponseReader;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.ical.ICalHelper;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.DomainServiceRegistry;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.MhpObjectFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.SystemsDataLayerFactory;
import com.agilex.healthcare.mobilehealthplatform.utils.PatientIdentifierHelper;
import com.agilex.healthcare.utility.AppointmentComparator;
import com.agilex.healthcare.utility.AppointmentReverseComparator;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.vamf.cdw.appointment.CdwAppointmentDataLayer;

public class AppointmentDataService {

	private MhpObjectFactory mhpObjectFactory = null;
	private Router router = new Router();
	private CdwAppointmentDataLayer dataLayer;

	public AppointmentDataService() {
		this.mhpObjectFactory = MhpObjectFactory.getInstance();
		dataLayer = this.mhpObjectFactory.getCdwAppointmentDataLayer();
	}

	public Appointments getPatientAppointments(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		return getPatientAppointments(patientIdentifier, dateFilter, ScopeFilter.getInstanceForLongitudinalScope());
	}
	
	public Appointments getPatientAppointmentsWithDetails(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		return getPatientAppointmentsWithDetails(patientIdentifier, dateFilter, ScopeFilter.getInstanceForLongitudinalScope());
	}
	
	public Appointments getPatientAppointments(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		/*RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.appointment).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).forDateFilter(dateFilter).excludeDetails().build();
		ResponseMessage responseMessage = router.execute(request);

		Appointments appointments = PatientDataResponseReader.<Appointments, Appointment> fromResponse(responseMessage).getDataListNoNull(Appointments.class);*/
		PatientIdentifiers icnPatientIdentifiers = PatientIdentifierHelper.translatePatientIdentifier(patientIdentifier, "ICN");
		if(NullChecker.isNullish(icnPatientIdentifiers))
			return new Appointments();
		Appointments appointments = dataLayer.fetchAppointmentsByPatient(icnPatientIdentifiers.get(0), dateFilter);
		PatientIdentifierHelper.updatePatientIdentifier(appointments, patientIdentifier);
		return appointments;
	}
	
	
	
	public Appointments getPatientAppointmentsWithDetails(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.appointment).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).forDateFilter(dateFilter).includeDetails().build();
		ResponseMessage responseMessage = router.execute(request);

		Appointments appointments = PatientDataResponseReader.<Appointments, Appointment> fromResponse(responseMessage).getDataListNoNull(Appointments.class);
		return appointments;
	}
	
	public Appointments fetchMentalHealthAppointments(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(Domain.mentalHealthAppointment).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).forDateFilter(dateFilter).build();
		ResponseMessage responseMessage = router.execute(request);
		
		Appointments appointments = PatientDataResponseReader.<Appointments, Appointment>fromResponse(responseMessage).getDataListNoNull(Appointments.class);
		return appointments;
	}

	public Appointment getPatientAppointment(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveSingleById().forDomain(Domain.appointment).forPatientIdentifier(patientIdentifier).forDataIdentifier(dataIdentifier).build();
		ResponseMessage responseMessage = router.execute(request);
		Appointment appointment = PatientDataResponseReader.<Appointments, Appointment> fromResponse(responseMessage).getDataItem();
		return appointment;
	}

	public Appointments getProviderAppointments(String providerId, DateFilter datefilter) {
		// to explain why getProvider is different from getPatient =>
		// non-patient retrieves currently do not go through the "retriever"

		DomainServiceRegistry registry = new DomainServiceRegistry(mhpObjectFactory);
		SystemsDataLayerFactory<AppointmentDataLayer> system = registry.getSingleDataLayerFactoriesByScopeFilter(ScopeFilter.getInstanceForLongitudinalScope(), Domain.appointment);

		return system.getDataLayerInstance().getProviderAppointments(providerId, datefilter);
	}
	
	public ProviderNote fetchClinicNotes(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier, ScopeFilter scopeFilter) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveSingleById().forDomain(Domain.clinicNote).forDataIdentifier(dataIdentifier).forPatientIdentifier(patientIdentifier).build();
		ResponseMessage response = router.execute(request);
		
		return PatientDataResponseReader.<ProviderNotes, ProviderNote> fromResponse(response).getDataItem();
	}

	public Appointments getClinicAppointments(String providerId, DateFilter datefilter) {
		DomainServiceRegistry registry = new DomainServiceRegistry(mhpObjectFactory);
		SystemsDataLayerFactory<AppointmentDataLayer> system = registry.getSingleDataLayerFactoriesByScopeFilter(ScopeFilter.getInstanceForLongitudinalScope(), Domain.appointment);

		return system.getDataLayerInstance().getClinicAppointments(providerId, datefilter);
	}

	public Appointments sortByDate(Appointments appointments, boolean isDescending) {
		if (isDescending) {
			Collections.sort(appointments, new AppointmentComparator());
		} else {
			Collections.sort(appointments, new AppointmentReverseComparator());
		}
		return appointments;
	}

	public String createIcsFromAppointments(Appointments appointments) {
		String ical = ICalHelper.transformAppointmentsToIcal(appointments).toString();
		return ical;
	}

	

}
