package com.agilex.healthcare.mobilehealthplatform.datalayer;

import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.RequestMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.ResponseMessage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.Router;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataEditRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataFetchRequestBuilder;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dataretriever.router.patientdata.PatientDataResponseReader;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDataCollection;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.validator.Validator;

public abstract class JournalEventsDataService<Ts extends PatientDataCollection<T>, T extends PatientData> {
	private Router router;
	private String domain;
	private Class<Ts> datalistType;
	private Validator<T> validator;

	public JournalEventsDataService(String domain, Class<Ts> datalistType, Validator<T> validator) {
		this.router = new Router();
		this.domain = domain;
		this.datalistType = datalistType;
		this.validator = validator;
	}

	public JournalEventsDataService(Router router, String domain, Class<Ts> datalistType, Validator<T> validator) {
		this.router = router;
		this.domain = domain;
		this.datalistType = datalistType;
		this.validator = validator;
	}

	public Ts fetchDataList(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveList().forDomain(domain).forPatientIdentifier(patientIdentifier).forScopeFilter(scopeFilter).forDateFilter(dateFilter).build();
		ResponseMessage responseMessage = router.execute(request);
		Ts datalist = PatientDataResponseReader.<Ts, T> fromResponse(responseMessage).getDataListNoNull(datalistType);
		datalist.sortDescending("date");
		return datalist;
	}

	public T fetchData(PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier) {
		RequestMessage request = PatientDataFetchRequestBuilder.forRetrieveSingleById().forDomain(domain).forPatientIdentifier(patientIdentifier).forDataIdentifier(dataIdentifier).build();
		ResponseMessage responseMessage = router.execute(request);
		return PatientDataResponseReader.<Ts, T> fromResponse(responseMessage).getDataItem();
	}

	public T createData(T data, ScopeFilter scopeFilter) {
		validateBeforeSave(data);
        addSurrogate(data);
		RequestMessage request = PatientDataEditRequestBuilder.forCreate().forDomain(domain).forScope(scopeFilter).forData(data).build();
		ResponseMessage responseMessage = router.execute(request);
		return PatientDataResponseReader.<Ts, T> fromResponse(responseMessage).getDataItem();
	}

	public T updateData(T data) {
		validateBeforeSave(data);
        addSurrogate(data);
        RequestMessage request = PatientDataEditRequestBuilder.forUpdate().forDomain(domain).forData(data).build();
		ResponseMessage responseMessage = router.execute(request);
		return PatientDataResponseReader.<Ts, T> fromResponse(responseMessage).getDataItem();
	}

	public void deleteData(T data) {
        addSurrogate(data);
        RequestMessage request = PatientDataEditRequestBuilder.forDelete().forDomain(domain).forData(data).build();
		router.execute(request);
	}

	private void validateBeforeSave(T data) {
		if (validator != null) {
			ValidationResult<T> validationResult = validator.validate(data);
			if (!validationResult.isValid()) {
				throwValidationException(validationResult);
			}
		}
	}

    private void addSurrogate(T data) {
        MhpUserFactory.addSurrogateToPatientData(data);
    }

    protected abstract void throwValidationException(ValidationResult<T> validationResult);
}
