package com.agilex.healthcare.mobilehealthplatform.security;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import junit.framework.Assert;

import org.junit.Test;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.web.FilterInvocation;

import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;

//@ContextConfiguration(locations = { "classpath:ApplicationContext.xml"})
//@RunWith(SpringJUnit4ClassRunner.class)
public class PatientUserVoterTest extends AbstractVoterTest{
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientUserVoterTest.class);

    private String patientUserVoterSupportedAttribute = "IS_PATIENT_OR_ADVOCATE";

    @Test
	public void supportsDesiredConfigAccess(){
		Assert.assertTrue(supportsConfigAttribute("IS_PATIENT_OR_ADVOCATE", new PatientUserVoter<FilterInvocation>()));
	}

	@Test
	public void doesNotSupportsInvalidConfigAccess(){
		Assert.assertFalse(supportsConfigAttribute(Roles.ROLE_CONSUMER, new PatientUserVoter<FilterInvocation>()));
	}
	
	@Test
	public void nonPatientAbstains(){
		PatientUserVoter<FilterInvocation> voter = new PatientUserVoter<FilterInvocation>();
		List<GrantedAuthority> authorities = createStaffAuthorities();

		AppUser principal = createPrincipal("gallow.younger", new PatientIdentifier(defaultAssigningAuthority, defaultTargetPatientId), authorities);
		
		Object credentials = null;

		Authentication authentication = createAuthentication(authorities, principal, credentials);

		//Use knowledge of currently hard-coded list of patients that are in pilot program
		FilterInvocation fi = createFilterInvocation(defaultAssigningAuthority, defaultTargetPatientId);
		
		Collection<ConfigAttribute> attributes = new ArrayList<ConfigAttribute>();
		
		int result = voter.vote(authentication, fi, attributes);
		
		Assert.assertEquals(AccessDecisionVoter.ACCESS_ABSTAIN, result);
	}
	
	@Test
	public void patientWithValidDSLogonShouldHaveAccess(){
		PatientUserVoter<FilterInvocation> voter = new PatientUserVoter<FilterInvocation>();
		List<GrantedAuthority> authorities = createPatientAuthorities();

		String userName = "noidentity.user";
		String assigningAuthority = "EDIPI";
		String uniqueId = "noidentity";
		AppUser principal = createPrincipal(userName, new PatientIdentifier(assigningAuthority, uniqueId), authorities);
		setPermittedPatient(principal, assigningAuthority, uniqueId);
		Object credentials = null;

		Authentication authentication = createAuthentication(authorities, principal, credentials);

		//Use knowledge of currently hard-coded list of patients that are in pilot program
		FilterInvocation fi = createFilterInvocation(assigningAuthority, uniqueId);
		fi.getRequest().getSession().setAttribute("PatientUserVoter.hasAcceptedRightOfAccess", Boolean.TRUE);
		
		Collection<ConfigAttribute> attributes = createSupportedAttributes();
		
		int result = voter.vote(authentication, fi, attributes);
		
        Assert.assertEquals(AccessDecisionVoter.ACCESS_GRANTED, result);
	}
	
	
	//Eventually this will need to take into account the advocates
	@Test
	public void patientViewingOtherPatientDenies(){
		PatientUserVoter<FilterInvocation> voter = new PatientUserVoter<FilterInvocation>();
		List<GrantedAuthority> authorities = createPatientAuthorities();

        AppUser principal = createPatientPrincipal(authorities);
		Object credentials = null;

		Authentication authentication = createAuthentication(authorities, principal, credentials);

		//Use knowledge of currently hard-coded list of patients that are in pilot program
		FilterInvocation fi = createFilterInvocation(defaultAssigningAuthority, defaultTargetPatientId + "B");  //add to end of identifier to ensure it is different

        Collection<ConfigAttribute> attributes = createSupportedAttributes();

		int result = voter.vote(authentication, fi, attributes);
		
		Assert.assertEquals(AccessDecisionVoter.ACCESS_DENIED, result);
	}

	@Test
	public void patientViewingSelfGrants(){
        logger.debug("--------------- PATIENT VIEWING SELF GRANTS ---------------------");
        int result = runPatientViewingSelf(true);
		Assert.assertEquals(AccessDecisionVoter.ACCESS_GRANTED, result);
        logger.debug("--------------- PATIENT VIEWING SELF GRANTS ---------------------");
	}

    @Test
    public void rightOfAccessFalseDenies(){
        int result = runPatientViewingSelf(false);
        Assert.assertEquals(AccessDecisionVoter.ACCESS_DENIED, result);
    }

    private int runPatientViewingSelf(boolean rightOfAccessAccepted) {
        PatientUserVoter<FilterInvocation> voter = new PatientUserVoter<FilterInvocation>();
        List<GrantedAuthority> authorities = createPatientAuthorities();
        AppUser principal = createPatientPrincipal(authorities);

        //if right of access false, need to change user to one that doesn have ROA set in RightOfAccessDataLayer (Mock or Repo)
        if (!rightOfAccessAccepted){
            principal.getMhpUser().getUserIdentifier().setUniqueId("ABC123");
            principal.getMhpUser().getPatient().getPatientIdentifier().setUniqueId("ABC123");
        }
        principal.getMhpUser().setRightOfAccessAccepted(rightOfAccessAccepted);

        Object credentials = null;

        Authentication authentication = createAuthentication(authorities, principal, credentials);

        //Use knowledge of currently hard-coded list of patients that are in pilot program
        FilterInvocation fi = createFilterInvocation(defaultAssigningAuthority, defaultTargetPatientId);

        Collection<ConfigAttribute> attributes = createSupportedAttributes();

        return voter.vote(authentication, fi, attributes);
    }

    private AppUser createPatientPrincipal(List<GrantedAuthority> authorities) {
		AppUser principal = createPrincipal("gallow.younger", new PatientIdentifier(defaultAssigningAuthority, defaultTargetPatientId), authorities);
        setPermittedPatient(principal, defaultAssigningAuthority, defaultTargetPatientId);
        return principal;
    }

    @Test
    public void advocateViewingPermittedPatientGrants(){
        logger.debug("--------------- ADVOCATE VIEWING SELF GRANTS ---------------------");
        PatientUserVoter<FilterInvocation> voter = new PatientUserVoter<FilterInvocation>();
        List<GrantedAuthority> authorities = createPatientAuthorities();

		AppUser principal = createPrincipal("gallow.younger", new PatientIdentifier(defaultAssigningAuthority, defaultTargetPatientId), authorities);
        setPermittedPatient(principal, defaultAssigningAuthority, defaultTargetPatientId);
        Object credentials = null;

        Authentication authentication = createAuthentication(authorities, principal, credentials);

        //Use knowledge of currently hard-coded list of patients that are in pilot program
        FilterInvocation fi = createFilterInvocation(defaultAssigningAuthority, defaultTargetPatientId);

        Collection<ConfigAttribute> attributes = createSupportedAttributes();

        int result = voter.vote(authentication, fi, attributes);

        Assert.assertEquals(AccessDecisionVoter.ACCESS_GRANTED, result);
        logger.debug("--------------- ADVOCATE VIEWING SELF GRANTS ---------------------");
    }

    @Test
    public void advocateViewingNonPermittedPatientDenies(){
        PatientUserVoter<FilterInvocation> voter = new PatientUserVoter<FilterInvocation>();
        List<GrantedAuthority> authorities = createPatientAuthorities();

		AppUser principal = createPrincipal("gallow.younger", new PatientIdentifier(defaultAssigningAuthority, defaultTargetPatientId), authorities);
        setPermittedPatient(principal, defaultAssigningAuthority, alternatePatientId);
        Object credentials = null;

        Authentication authentication = createAuthentication(authorities, principal, credentials);

        //Use knowledge of currently hard-coded list of patients that are in pilot program
        FilterInvocation fi = createFilterInvocation(defaultAssigningAuthority, defaultTargetPatientId);

        Collection<ConfigAttribute> attributes = createSupportedAttributes();

        int result = voter.vote(authentication, fi, attributes);

        Assert.assertEquals(AccessDecisionVoter.ACCESS_DENIED, result);
    }

    private Collection<ConfigAttribute> createSupportedAttributes() {
        Collection<ConfigAttribute> attributes = new ArrayList<ConfigAttribute>();
        ConfigAttribute attribute = new ConfigAttribute() {
			private static final long serialVersionUID = 9208629368448580596L;

			@Override
            public String getAttribute() {
                return patientUserVoterSupportedAttribute;
            }
        }                                             ;
        attributes.add(attribute);
        return attributes;
    }

    private void setPermittedPatient(AppUser principal, String defaultAssigningAuthority1, String targetPatientId) {
        Patient advocatePatientTarget = new Patient();
        PatientIdentifier patientIdentifier = new PatientIdentifier();
        patientIdentifier.setAssigningAuthority(defaultAssigningAuthority1);
        patientIdentifier.setUniqueId(targetPatientId);
        advocatePatientTarget.setPatientIdentifier(patientIdentifier);
        principal.getMhpUser().setPatient(advocatePatientTarget);
    }
}
