package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.List;

import javapns.Push;
import javapns.communication.exceptions.CommunicationException;
import javapns.communication.exceptions.KeystoreException;
import javapns.devices.Device;
import javapns.devices.exceptions.InvalidDeviceTokenFormatException;
import javapns.notification.PushNotificationPayload;
import javapns.notification.PushedNotification;
import javapns.notification.ResponsePacket;
import javapns.notification.exceptions.PayloadMaxSizeExceededException;

import org.json.JSONException;
import org.junit.Ignore;
import org.junit.Test;

public class JavaPNSLearningTest {
	private static final String KEYSTORE_PATH = "src/main/resources/launchpad-va-enterprise_keystore-production.p12";
	private static final String KEYSTORE_PASSWORD = "Agilexadmin99$";
	private static final boolean USE_PRODUCTION_APNS = true;
	private static final String[] DEVICE_TOKENS = { "0a701d904f7a06b52904cefe3ccc3df8f0a2e88bea9ab62939b5d3b2dd52710b" };
	private static final String DEVICE_TOKEN = DEVICE_TOKENS[0];

	private static final String DEFAULT_MESSAGE = "test message";
	private static final String DEFAULT_SOUND = "default";
	private static final String APPLICATION_KEY = "app";
	private static final String NOTIFICATION_APPLICATION_VALUE = "NTFN";

	@Test
	public void testBasicAlertPush() {
		try {
			List<PushedNotification> notifications = Push.alert(DEFAULT_MESSAGE, KEYSTORE_PATH, KEYSTORE_PASSWORD, USE_PRODUCTION_APNS, DEVICE_TOKEN);

			printPushedNotificationResults(notifications);
		} catch (KeystoreException e) {
			failTest("Check keystore configurations", e);
		} catch (CommunicationException e) {
			failTest("Error communicating with Apple servers", e);
		}
	}

	@Test
	public void testComplexPush() {
		try {
			PushNotificationPayload payload = PushNotificationPayload.complex();
			payload.addAlert(DEFAULT_MESSAGE);
			payload.addSound(DEFAULT_SOUND);
			payload.addCustomDictionary(APPLICATION_KEY, NOTIFICATION_APPLICATION_VALUE);

			/*
			 * Example custom payload
			 * {"app":"NTFN","aps":{"sound":"default","alert":"test message"}}
			 */

			int numberOfThreads = 1;

			List<PushedNotification> notifications = Push.payload(payload, KEYSTORE_PATH, KEYSTORE_PASSWORD, USE_PRODUCTION_APNS, numberOfThreads, DEVICE_TOKENS);

			printPushedNotificationResults(notifications);
		} catch (KeystoreException e) {
			failTest("Check keystore configurations", e);
		} catch (CommunicationException e) {
			failTest("Error communicating with Apple servers", e);
		} catch (JSONException e) {
			failTest("Error building custom payload", e);
		} catch (Exception e) {
			failTest("Error sending custom payload", e);
		}
	}

	@Test
	public void testExceedMaxSize() {
		String longMessage = "12345678901234567890123456789012345678901234567890" + "12345678901234567890123456789012345678901234567890" + "12345678901234567890123456789012345678901234567890"
				+ "12345678901234567890123456789012345678901234567890" + "12345678901234567890123456789012345678901234567890" + "12345678901234567890123456789012345678901234567890"
				+ "12345678901234567890123456789012345678901234567890" + "12345678901234567890123456789012345678901234567890" + "eom";

		try {
			List<PushedNotification> notifications = Push.alert(longMessage, KEYSTORE_PATH, KEYSTORE_PASSWORD, USE_PRODUCTION_APNS, DEVICE_TOKENS);

			assertFalse(notifications.get(0).isSuccessful());
			assertTrue(notifications.get(0).getException() instanceof PayloadMaxSizeExceededException);
		} catch (KeystoreException e) {
			failTest("Check keystore configurations", e);
		} catch (CommunicationException e) {
			failTest("Error communicating with Apple servers", e);
		}
	}

	@Test
	@Ignore
	public void testFeedbackService() {
		try {
			List<Device> inactiveDevices = Push.feedback(KEYSTORE_PATH, KEYSTORE_PASSWORD, USE_PRODUCTION_APNS);

			for (Device device : inactiveDevices) {
				System.out.println(String.format("Inactive device id [%s] with token [%s]", device.getDeviceId(), device.getToken()));
			}
		} catch (KeystoreException e) {
			failTest("Keystore exception!", e);
		} catch (CommunicationException e) {
			failTest("Communication exception!  Unable to talk to Apple Servers.", e);
		}
	}

	private void printPushedNotificationResults(List<PushedNotification> notifications) {
		for (PushedNotification notification : notifications) {
			if (notification.isSuccessful()) {
				System.out.println(String.format("Notification to device token [%s] was successful", notification.getDevice().getToken()));
			} else {
				System.out.println(String.format("Notification to device token [%s] failed", notification.getDevice().getToken()));

				// Find out more about what the problem was
				notification.getException().printStackTrace();

				// If the problem was an error-response packet returned by Apple
				// or if the device token is in an invalid format
				ResponsePacket theErrorResponse = notification.getResponse();
				if (theErrorResponse != null || notifications.get(0).getException() instanceof InvalidDeviceTokenFormatException) {
					System.out.println("Remove invalid token" + notification.getDevice().getToken());
				}
			}
		}
	}

	private void failTest(String message, Exception e) {
		e.printStackTrace();
		fail(message);
	}
}
