package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.test.SpringBasedIntegration;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperties;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperty;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestion;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestionChoice;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestionChoices;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponse;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponses;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResultReport;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;

public class PTSDReportGeneratorTest extends SpringBasedIntegration{
	private PTSDReportGenerator reportGenerator = new PTSDReportGenerator();
	private PTSDReportSettings reportSettings = new PTSDReportSettings();

	@Test
	public void testSimpleReport() {
		AssessmentResult result = generateMockAssessmentResult(2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);

		assertNotNull(resultReport);
		assertNotNull(reportText);
		assertTrue(reportText.contains(reportSettings.endOfAllResultsText));
	}
	
	@Test
	public void testReportWithNoPriorScore() {
		AssessmentResult result = generateMockAssessmentResult(2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(resultReport);
		assertTrue(reportText.contains(reportSettings.noPreviousResultText));
	}

	@Test
	public void testReportWithSameScore() {
		AssessmentResult result = generateMockAssessmentResult(2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResult previousResult = generatePreviousResult(2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, previousResult);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(resultReport);
		assertTrue(reportText.contains(reportSettings.sameScoreText));
	}

	@Test
	public void testReportWithIncreaseInScore() {
		AssessmentResult previousResult = generatePreviousResult(2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResult result = generateMockAssessmentResult(3, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, previousResult);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(resultReport);
		assertTrue(reportText.contains(reportSettings.increasedScoreText));
	}

	@Test
	public void testReportWithDecreaseInScore() {
		AssessmentResult previousResult = generatePreviousResult(3, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResult result = generateMockAssessmentResult(2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, previousResult);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(resultReport);
		assertTrue(reportText.contains(reportSettings.decreasedScoreText));
	}

	@Test
	public void testReportWithLowestScore() {
		AssessmentResult result = generateMockAssessmentResult(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		System.out.println(reportText);
		
		assertNotNull(resultReport);
		assertTrue(reportText.contains(reportSettings.lowestScoreText));
	}

	@Test
	public void testReportWithLowScore() {
		AssessmentResult result = generateMockAssessmentResult(2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(resultReport);
		assertTrue(reportText.contains(reportSettings.lowScoreText));
	}

	@Test
	public void testReportWithMediumScore() {
		AssessmentResult result = generateMockAssessmentResult(3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(resultReport);
		assertTrue(reportText.contains(reportSettings.mediumScoreText));
	}

	@Test
	public void testReportWithHighScore() {
		AssessmentResult result = generateMockAssessmentResult(5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5);
		AssessmentResultReport resultReport = reportGenerator.generateReport(result, null);
		String reportText = resultReport.getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		
		assertNotNull(resultReport);
		assertTrue(reportText.contains(reportSettings.highScoreText));
	}

	private AssessmentResult generatePreviousResult(int... responsePointValues) {
		AssessmentResult previousResult = generateMockAssessmentResult(responsePointValues);
		Date yesterday = DateHelper.getYesterday();
		previousResult.setDateTaken(yesterday);
		return previousResult;
	}

	private AssessmentResult generateMockAssessmentResult(int... responsePointValues) {
		AssessmentResult mockResult = new AssessmentResult();
		mockResult.setAssessmentId("testId");
		mockResult.setPatientIdentifier("mock-aa", "INT-TEST-MOCK");
		mockResult.setScoringAlgorithm(AssessmentCode.SCORING_ALGORITHM_ZARIT);
		mockResult.setResponses(generateMockResponses(responsePointValues));
		int totalScore = 0;
		for (int responsePointValue : responsePointValues) {
			totalScore += responsePointValue;
		}
		mockResult.setScore(totalScore);
		return mockResult;
	}

	private AssessmentResponses generateMockResponses(int... responsePointValues) {
		AssessmentResponses assessmentResponses = new AssessmentResponses();
		int questionNumber = 1;
		for (int responsePointValue : responsePointValues) {
			assessmentResponses = addResponse(assessmentResponses, responsePointValue, questionNumber);
			questionNumber++;
		}
		return assessmentResponses;
	}

	private AssessmentResponses addResponse(AssessmentResponses assessmentResponses, int pointValue, int questionNumber) {
		AssessmentResponse assessmentResponse = new AssessmentResponse();
		AssessmentQuestionChoices selectedChoices = new AssessmentQuestionChoices();
		AssessmentQuestionChoice selectedChoice = new AssessmentQuestionChoice();
		AssessmentProperties properties = new AssessmentProperties();
		AssessmentQuestion assessmentQuestion = new AssessmentQuestion();

		properties.add(new AssessmentProperty(AssessmentCode.RESULT_PROPERTY_POINT_VALUE, "" + pointValue));
		selectedChoice.setProperties(properties);
		selectedChoices.add(selectedChoice);
		assessmentResponse.setSelectedChoices(selectedChoices);

		assessmentQuestion.setQuestionNumber(questionNumber);
		assessmentResponse.setQuestion(assessmentQuestion);

		assessmentResponses.add(assessmentResponse);
		return assessmentResponses;
	}
}
