package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.util.Date;
import java.util.UUID;

import com.agilex.healthcare.mobilehealthplatform.test.SpringBasedIntegration;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponses;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResults;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;

public class AssessmentResultsDataServiceTest extends SpringBasedIntegration{
	private AssessmentResultsDataService dataservice = new AssessmentResultsDataService();
	private PatientIdentifier anonymousIdentifier = AssessmentCode.ANONYMOUS_RESULT_PATIENT_ID;
	private PatientIdentifier randomIdentifier = generateRandomPatientIdentifier();
	private ScopeFilter scope = ScopeFilter.getInstanceForLongitudinalScope();

	@Test
	public void saveAnonymousResultCorrectly() {
		saveResult(createAnonymousResult(anonymousIdentifier));
	}

	@Test
	public void saveAnonymousResultIncorrectly() {
		AssessmentResult savedResult = saveResult(createAnonymousResult(randomIdentifier));
		assertNotNull(savedResult);
		assertEquals(anonymousIdentifier, savedResult.getPatientIdentifier());

		savedResult = saveResult(createAnonymousResult(anonymousIdentifier));
		assertNotNull(savedResult);
		assertEquals(anonymousIdentifier, savedResult.getPatientIdentifier());

		savedResult = saveResult(createAnonymousResult(randomIdentifier));
		assertNotNull(savedResult);
		assertEquals(anonymousIdentifier, savedResult.getPatientIdentifier());
	}

	@Test
	public void saveNonAnonymousResultCorrectly() {
		AssessmentResult savedResult = saveResult(createNonAnonymousResult(randomIdentifier));
		assertNotNull(savedResult);
	}

	@Test
	public void saveDraftCorrectly() {
		AssessmentResult draft = createDraft(randomIdentifier);
		saveResult(draft);

		AssessmentResults retrievedDrafts = dataservice.fetchAssessmentDraftsByPatient(randomIdentifier, scope);
		assertEquals(1, retrievedDrafts.size());

		AssessmentResult retrievedDraft = retrievedDrafts.get(0);
		assertEquals(draft.getUniqueTitle(), retrievedDraft.getUniqueTitle());
	}

	private AssessmentResult saveResult(AssessmentResult assessmentResult) {
		return dataservice.saveResult(assessmentResult, scope);
	}

	public AssessmentResult createAnonymousResult(PatientIdentifier patientIdentifier) {
		AssessmentResult assessmentResult = createAndPopulateSampleResult(patientIdentifier);
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_ANONYMOUS);
		assessmentResult.setAssessmentId("testAnonymous");
		return assessmentResult;
	}

	public AssessmentResult createNonAnonymousResult(PatientIdentifier patientIdentifier) {
		AssessmentResult assessmentResult = createAndPopulateSampleResult(patientIdentifier);
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_BASIC);
		assessmentResult.setAssessmentId("strain");
		return assessmentResult;
	}

	public AssessmentResult createDraft(PatientIdentifier patientIdentifier) {
		AssessmentResult assessmentResult = createAndPopulateSampleResult(patientIdentifier);
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_BASIC);
		assessmentResult.setAssessmentId("strain");
		assessmentResult.setInProgress(true);
		return assessmentResult;
	}

	public AssessmentResult createAndPopulateSampleResult(PatientIdentifier patientIdentifier) {
		AssessmentResult assessmentResult = new AssessmentResult();
		assessmentResult.setUniqueTitle("testUniqueTitle");
		assessmentResult.setVersion("1.0");
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_ANONYMOUS);
		assessmentResult.setNotes("This assessment came from the DataServiceTest.");
		assessmentResult.setAssessmentId("testAssessmentId");
		assessmentResult.setPatientIdentifier(patientIdentifier);
		assessmentResult.setDateTaken(new Date());
		assessmentResult.setInProgress(false);
		assessmentResult.setScoringAlgorithm(AssessmentCode.SCORING_ALGORITHM_UNSCORED);
		assessmentResult.setWaitingPeriod(7);

		AssessmentResponses responses = new AssessmentResponses();
		assessmentResult.setResponses(responses);

		return assessmentResult;
	}

	public PatientIdentifier generateRandomPatientIdentifier() {
		return new PatientIdentifier("EDIPI", UUID.randomUUID().toString());
	}
}
