package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import java.util.*;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.restservice.exception.InvalidAppointmentRequestMessageException;
import com.agilex.healthcare.mobilehealthplatform.security.AppUser;
import com.agilex.healthcare.mobilehealthplatform.test.SpringBasedIntegration;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestPurposeOfVisit;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestType;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestVisitType;
import com.agilex.healthcare.mobilehealthplatform.patientcorrelation.PatientCorrelationMock;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import org.springframework.security.authentication.TestingAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.context.SecurityContextImpl;

import static org.junit.Assert.*;

public class AppointmentRequestMessagesDataServiceTest extends SpringBasedIntegration{
    private final static PatientIdentifier patientIdentifier = PatientCorrelationMock.GallowMockPatientIdentifier;

    private AppointmentRequestDataService dataservice;

    private List<String> pilotSites;
    private String haMode;

    //Values used for test SecurityContext
    private static final String TEST_USER_ID = "PATID26";
    private static final String TEST_CURRENT_LAST_NAME = "New     ";
    private static final String TEST_CURRENT_FIRST_NAME = "NewAldie";

    public AppointmentRequestMessagesDataServiceTest() {

        dataservice = new AppointmentRequestDataService();
        haMode = ModeHelper.isProviderMode() ? "provider" : "veteran";

        pilotSites = new LinkedList<String>();
        pilotSites.add("688");
        pilotSites.add("537");
        pilotSites.add("538");
        pilotSites.add("640");
    }

    private static void createSpringSecurityContext() {
        SecurityContext securityContext = new SecurityContextImpl();
        MhpUser user = new MhpUser();
        user.setId(TEST_USER_ID);
        user.setFirstName(TEST_CURRENT_FIRST_NAME);
        user.setLastName(TEST_CURRENT_LAST_NAME);
        AppUser appUser = new AppUser("testuser", "testpassword", new LinkedList<GrantedAuthority>(), user);
        Authentication authentication = new TestingAuthenticationToken(appUser, null);
        securityContext.setAuthentication(authentication);
        SecurityContextHolder.setContext(securityContext);
    }

    private static void setSecurityContextPatient(){
        AppUser appUser = (AppUser) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        MhpUser mhpUser = appUser.getMhpUser();

        Patient patient = new Patient();
        patient.getPatientIdentifier().setAssigningAuthority("EDIPI");
        patient.getPatientIdentifier().setUniqueId("D123401");
        patient.setFirstName("Gallow");
        patient.setLastName("Younger");
        patient.setDateOfBirth( new DOBDate("yyyyMMdd", "20100310"));
        patient.setSsn("111-11-1111");

        mhpUser.setPatient(patient);
    }

    @BeforeClass
    public static void init() {
        Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));

        createSpringSecurityContext();
        setSecurityContextPatient();
    }

    @AfterClass
    public static void teardown(){
        SecurityContextHolder.setContext(new SecurityContextImpl());
    }

    @Test
    public void retrieveAppointmentRequestMessages() {

        AppointmentRequest newRequest1 = createValidAppointmentRequest();
        newRequest1.setAppointmentDate(DateHelper.formatDate(DateHelper.getTomorrow()));
        AppointmentRequest savedAppointmentRequest1 = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());

        AppointmentRequestMessage appointmentRequestMessage =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        AppointmentRequestMessage savedRequestMessage;
        try{
            savedRequestMessage = dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
            AppointmentRequestMessages messages = dataservice.fetchAppointmentRequestMessages(savedAppointmentRequest1.getPatientIdentifier(), savedAppointmentRequest1.getDataIdentifier());
            assertNotNull(messages);
            assertEquals(1, messages.size());
            assertEquals(savedRequestMessage.getAppointmentRequestId(), messages.get(0).getAppointmentRequestId());


        }finally {
            if(savedAppointmentRequest1 != null){
                dataservice.deleteAppointmentRequest(savedAppointmentRequest1, ScopeFilter.getInstanceForLongitudinalScope());
            }
        }

    }


    @Test
    public void testMessageNotSavedIfAppointmentIsCancelled(){
        boolean didReceiveException = false;

        AppointmentRequest newRequest1 = createValidAppointmentRequest();
        newRequest1.setStatus("Cancelled");
        AppointmentRequest savedAppointmentRequest1 = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());

        AppointmentRequestMessage appointmentRequestMessage =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        try{
            dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
        }catch(InvalidAppointmentRequestMessageException e){
            assertEquals(e.getErrors().get(0).getErrorMessage(), "Appointment request messages are not allowed for cancelled appointment requests.");
            didReceiveException = true;
        }finally {
            if(savedAppointmentRequest1 != null){
                dataservice.deleteAppointmentRequest(savedAppointmentRequest1, ScopeFilter.getInstanceForLongitudinalScope());
            }

        }

        assertTrue(didReceiveException);

    }

    @Test
    public void testMessageNotSavedIfAppointmentDateHasPassed(){
        boolean didReceiveException = false;

        AppointmentRequest newRequest1 = createValidAppointmentRequest();
        newRequest1.setAppointmentDate(DateHelper.formatDate(DateHelper.getYesterday()));
        AppointmentRequest savedAppointmentRequest1 = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());

        AppointmentRequestMessage appointmentRequestMessage =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        try{
            dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
        }catch(InvalidAppointmentRequestMessageException e){
            assertEquals(e.getErrors().get(0).getErrorMessage(), "Appointment request messages are not allowed after appointment date has passed.");
            didReceiveException = true;
        }finally {
            if(savedAppointmentRequest1 != null){
                dataservice.deleteAppointmentRequest(savedAppointmentRequest1, ScopeFilter.getInstanceForLongitudinalScope());
            }

        }

        assertTrue(didReceiveException);

    }

    @Test
    public void testMessageSavedIfAppointmentDateIsInFuture(){

        AppointmentRequest newRequest1 = createValidAppointmentRequest();
        newRequest1.setAppointmentDate(DateHelper.formatDate(DateHelper.getTomorrow()));
        AppointmentRequest savedAppointmentRequest1 = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());

        AppointmentRequestMessage appointmentRequestMessage =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        AppointmentRequestMessage savedRequestMessage = null;
        try{
            savedRequestMessage = dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
        }finally {
            if(savedAppointmentRequest1 != null){
                dataservice.deleteAppointmentRequest(savedAppointmentRequest1, ScopeFilter.getInstanceForLongitudinalScope());
            }
        }

        assertNotNull(savedRequestMessage);

    }

    @Test
    public void testSenderCannotSendMessageUntilReplyEntered(){

        AppointmentRequest newRequest1 = createValidAppointmentRequest();
        newRequest1.setAppointmentDate(DateHelper.formatDate(DateHelper.getToday()));
        AppointmentRequest savedAppointmentRequest1 = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());

        AppointmentRequestMessage appointmentRequestMessage =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        AppointmentRequestMessage firstSavedMessage = null;
        boolean secondSaveFailed = false;
        try{
            firstSavedMessage = dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
            dataservice.saveAppointmentRequestMessage(appointmentRequestMessage); //consecutive messages without reply should fail

        }catch(InvalidAppointmentRequestMessageException e){
            assertEquals(e.getErrors().get(0).getErrorMessage(), "New appointment request message not allowed until reply received.");
            secondSaveFailed = true;
        }finally {
            if(savedAppointmentRequest1 != null){
                dataservice.deleteAppointmentRequest(savedAppointmentRequest1, ScopeFilter.getInstanceForLongitudinalScope());
            }
        }
        assertNotNull(firstSavedMessage);
        assertTrue(secondSaveFailed);

    }

    @Test
    public void testTwoMessageExchangesCanTakePlace(){

        AppointmentRequest newRequest1 = createValidAppointmentRequest();
        newRequest1.setAppointmentDate(DateHelper.formatDate(DateHelper.getToday()));
        AppointmentRequest savedAppointmentRequest1 = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());

        AppointmentRequestMessage appointmentRequestMessage =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        AppointmentRequestMessage reply =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        reply.setSenderId("11");
        AppointmentRequestMessage firstSavedMessage = null;
        AppointmentRequestMessage firstReplyMessage = null;
        AppointmentRequestMessage secondMessage = null;
        AppointmentRequestMessage secondReplyMessage = null;
        try{
            firstSavedMessage = dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
            reply.setMessageDateTime(new Date());
            firstReplyMessage = dataservice.saveAppointmentRequestMessage(reply); //consecutive messages without reply should fail
            appointmentRequestMessage.setMessageDateTime(new Date());
            secondMessage = dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
            reply.setMessageDateTime(new Date());
            secondReplyMessage = dataservice.saveAppointmentRequestMessage(reply);

        }finally {
            if(savedAppointmentRequest1 != null){
                dataservice.deleteAppointmentRequest(savedAppointmentRequest1, ScopeFilter.getInstanceForLongitudinalScope());
            }
        }
        assertNotNull(firstSavedMessage);
        assertNotNull(firstReplyMessage);
        assertNotNull(secondMessage);
        assertNotNull(secondReplyMessage);
    }


    @Test
    public void testNoSenderCanSendMoreThanTwoMessages(){

        AppointmentRequest newRequest1 = createValidAppointmentRequest();
        newRequest1.setAppointmentDate(DateHelper.formatDate(DateHelper.getToday()));
        AppointmentRequest savedAppointmentRequest1 = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());

        AppointmentRequestMessage appointmentRequestMessage =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        AppointmentRequestMessage reply =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        reply.setSenderId("11");
        AppointmentRequestMessage firstSavedMessage = null;
        AppointmentRequestMessage firstReplyMessage = null;
        AppointmentRequestMessage secondMessage = null;
        AppointmentRequestMessage secondReplyMessage = null;
        AppointmentRequestMessage thirdMessage = null;
        boolean thirdMessageTriggeredException = false;
        try{
            firstSavedMessage = dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
            reply.setMessageDateTime(new Date());
            firstReplyMessage = dataservice.saveAppointmentRequestMessage(reply); //consecutive messages without reply should fail
            appointmentRequestMessage.setMessageDateTime(new Date());
            secondMessage = dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
            reply.setMessageDateTime(new Date());
            secondReplyMessage = dataservice.saveAppointmentRequestMessage(reply);
            appointmentRequestMessage.setMessageDateTime(new Date());
            thirdMessage = dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);

        }catch(InvalidAppointmentRequestMessageException e){
            assertEquals(e.getErrors().get(0).getErrorMessage(), "Maximum allowed number of messages for this appointment request reached.");
            thirdMessageTriggeredException = true;
        }finally {
            if(savedAppointmentRequest1 != null){
                dataservice.deleteAppointmentRequest(savedAppointmentRequest1, ScopeFilter.getInstanceForLongitudinalScope());
            }
        }
        assertNotNull(firstSavedMessage);
        assertNotNull(firstReplyMessage);
        assertNotNull(secondMessage);
        assertNotNull(secondReplyMessage);
        assertNull(thirdMessage);
        assertTrue(thirdMessageTriggeredException);
    }


    @Test
    public void testMessageOver100CharactersTriggersError(){

        AppointmentRequest newRequest1 = createValidAppointmentRequest();
        newRequest1.setAppointmentDate(DateHelper.formatDate(new Date()));
        AppointmentRequest savedAppointmentRequest1 = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());

        AppointmentRequestMessage appointmentRequestMessage =  createValidAppointmentRequestMessage(savedAppointmentRequest1);
        appointmentRequestMessage.setMessageText(createMessageOver100Characters());
        AppointmentRequestMessage firstSavedMessage = null;
        boolean over100CharactersErrorDetected = false;
        try{
            firstSavedMessage = dataservice.saveAppointmentRequestMessage(appointmentRequestMessage);
        }catch(InvalidAppointmentRequestMessageException e){
            assertEquals(e.getErrors().get(0).getErrorMessage(), "Appointment request message is required and must be between 1 and 100 characters.");
            over100CharactersErrorDetected = true;
        }finally {
            if(savedAppointmentRequest1 != null){
                dataservice.deleteAppointmentRequest(savedAppointmentRequest1, ScopeFilter.getInstanceForLongitudinalScope());
            }
        }
        assertNull(firstSavedMessage);
        assertTrue(over100CharactersErrorDetected);

    }


    private AppointmentRequest createValidAppointmentRequest() {
        String appointmentType = AppointmentRequestType.MENTAL_HEALTH.getName();
        String email = "test@agilex.com";
        String phoneNumber = "+1 (555) 555-5555 x55.55";
        String visitType = AppointmentRequestVisitType.OFFICE_VISIT.getName();
        String purposeOfVisit = AppointmentRequestPurposeOfVisit.OTHER.getName();
        String otherPurposeOfVisit = "Other purpose of visit specified test";
        String providerId = "PROV1";
        boolean secondRequest = false;

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 7);
        String optionDate1 = DateHelper.formatDate(cal.getTime());
        String optionTime1 = "AM";
        cal.add(Calendar.DATE, 1);
        String optionDate2 = DateHelper.formatDate(cal.getTime());
        String optionTime2 = "PM";
        cal.add(Calendar.DATE, 2);
        String optionDate3 = DateHelper.formatDate(cal.getTime());
        String optionTime3 = "AM";
        HashSet<String> bestTimetoCall = new HashSet<String>();
        bestTimetoCall.add("9 AM - 11 AM");
        bestTimetoCall.add("11 AM - 1 PM");
        String textMessagingPhoneNumber = "555-222-3242";
        boolean textMessagingAllowed = true;
        String providerName = "John.     ";

        String facilityAddress = "test address";
        String facilityCity = "test city";
        String facilityState = "test state";
        String facilityCode = "537";
        String facilityName = "Jesse Brown VAMC";
        String facilityParentSiteCode = "537";
        String facilityType = "test facility type";

        Facility facility = new Facility();
        facility.setAddress(facilityAddress);
        facility.setCity(facilityCity);
        facility.setFacilityCode(facilityCode);
        facility.setName(facilityName);
        facility.setParentSiteCode(facilityParentSiteCode);
        facility.setState(facilityState);
        facility.setType(facilityType);

        AppointmentRequest a = new AppointmentRequest();

        a.setPatientIdentifier(patientIdentifier);
        a.setAppointmentType(appointmentType);
        a.setVisitType(visitType);
        a.setEmail(email);
        a.setPhoneNumber(phoneNumber);
        a.setOptionDate1(optionDate1);
        a.setOptionTime1(optionTime1);
        a.setOptionDate2(optionDate2);
        a.setOptionTime2(optionTime2);
        a.setOptionDate3(optionDate3);
        a.setOptionTime3(optionTime3);
        a.setFacility(facility);
        a.setBestTimetoCall(bestTimetoCall);
        a.setPurposeOfVisit(purposeOfVisit);
        a.setOtherPurposeOfVisit(otherPurposeOfVisit);
        a.setStatus(AppointmentRequestStatus.SUBMITTED.getName());
        a.setProviderId(providerId);
        a.setLastUpdatedDate(new Date());
        a.setSecondRequest(secondRequest);
        a.setTextMessagingAllowed(textMessagingAllowed);
        a.setTextMessagingPhoneNumber(textMessagingPhoneNumber);
        a.setPatientIdentifier(patientIdentifier);
        a.setCreatedDate(new Date());
        a.setProviderName(providerName);

        return a;
    }

    private AppointmentRequestMessage createValidAppointmentRequestMessage(AppointmentRequest appointmentRequest){

        AppointmentRequestMessage aptRequestMessage = new AppointmentRequestMessage();
        aptRequestMessage.setMessageDateTime(DateHelper.getToday());
        aptRequestMessage.setAppointmentRequestId(appointmentRequest.getAppointmentRequestId());
        aptRequestMessage.setSenderId("1");
        aptRequestMessage.setMessageText("Hello");
        aptRequestMessage.setPatientIdentifier(appointmentRequest.getPatientIdentifier());
        aptRequestMessage.setDataIdentifier(appointmentRequest.getDataIdentifier());

        return aptRequestMessage;

    }

    private String createMessageOver100Characters(){
        return createStringOfGivenLength(101);
    }

    private String createStringOfGivenLength(int stringLength){
        char fillerChar = 'a';
        StringBuilder stringBuilder = new StringBuilder();

        for (int i = 0; i < stringLength ; i++){
            stringBuilder.append(fillerChar);
        }
        return stringBuilder.toString();
    }
}
