package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.*;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.security.AppUser;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.test.SpringBasedIntegration;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestPurposeOfVisit;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestType;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestVisitType;
import com.agilex.healthcare.mobilehealthplatform.patientcorrelation.PatientCorrelationMock;
import com.agilex.healthcare.mobilehealthplatform.restservice.exception.InvalidAppointmentRequestException;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;
import org.springframework.security.authentication.TestingAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.context.SecurityContextImpl;

public class AppointmentRequestDataServiceTest extends SpringBasedIntegration{
	private final static PatientIdentifier patientIdentifier = PatientCorrelationMock.GallowMockPatientIdentifier;
	private final static PatientIdentifier patientIdentifierWithNoData = PatientCorrelationMock.HealthyEnterprisePatientIdentifier;
	
	private AppointmentRequestDataService dataservice;
	
	private AppointmentRequest appointmentRequest;
	private AppointmentRequest persistedAppointmentRequest;
	private DataIdentifier dataIdentifier;
	private DateFilter dateFilter;

	private List<String> pilotSites;
	private String haMode;

    //Values used for test SecurityContext
    private static final String TEST_USER_ID = "PATID26";
    private static final String TEST_CURRENT_LAST_NAME = "New     ";
    private static final String TEST_CURRENT_FIRST_NAME = "NewAldie";

	public AppointmentRequestDataServiceTest() {
		appointmentRequest = createValidAppointmentRequest();
		persistedAppointmentRequest = createValidAppointmentRequest();
		dataIdentifier = new DataIdentifier();
		dataIdentifier.setSystemId("var");
		dataIdentifier.setUniqueId("1");
		persistedAppointmentRequest.setDataIdentifier(dataIdentifier);
		dateFilter = DateFilterFactory.createEmptyFilter();

		dataservice = new AppointmentRequestDataService();
		haMode = ModeHelper.isProviderMode() ? "provider" : "veteran";
		
		pilotSites = new LinkedList<String>();
		pilotSites.add("688");
		pilotSites.add("537");
		pilotSites.add("538");
		pilotSites.add("640");
	}

    private static void createSpringSecurityContext() {
        SecurityContext securityContext = new SecurityContextImpl();
        MhpUser user = new MhpUser();
        user.setId(TEST_USER_ID);
        user.setFirstName(TEST_CURRENT_FIRST_NAME);
        user.setLastName(TEST_CURRENT_LAST_NAME);
        AppUser appUser = new AppUser("testuser", "testpassword", new LinkedList<GrantedAuthority>(), user);
        Authentication authentication = new TestingAuthenticationToken(appUser, null);
        securityContext.setAuthentication(authentication);
        SecurityContextHolder.setContext(securityContext);
    }

    private static void setSecurityContextPatient(){
        AppUser appUser = (AppUser) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        MhpUser mhpUser = appUser.getMhpUser();

        Patient patient = new Patient();
        patient.getPatientIdentifier().setAssigningAuthority("EDIPI");
        patient.getPatientIdentifier().setUniqueId("D123401");
        patient.setFirstName("Gallow");
        patient.setLastName("Younger");

        mhpUser.setPatient(patient);
    }

    @BeforeClass
	public static void init() {
		Assume.assumeTrue(ModeHelper.isMode("dev"));

        createSpringSecurityContext();
        setSecurityContextPatient();
	}

    @AfterClass
    public static void teardown(){
        SecurityContextHolder.setContext(new SecurityContextImpl());
    }

    @Test
    public void SecurityContextIsValid(){
        final MhpUser securityContext = MhpUserFactory.createFromSecurityContext();
        assertEquals("D123401", securityContext.getPatient().getPatientIdentifier().getUniqueId());
        assertEquals("EDIPI", securityContext.getPatient().getPatientIdentifier().getAssigningAuthority());

    }

	@Test
	public void validateAppointmentsRequestsAreSortedByLastUpdatedDateInDescendingOrder() {
		AppointmentRequests appointmentRequests = dataservice.getPatientAppointmentRequests(patientIdentifier, dateFilter, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(16, appointmentRequests.size());
		
		for(int i=0; i < appointmentRequests.size()-1; i++){
			if(appointmentRequests.get(i).getLastUpdatedDate().before(appointmentRequests.get(i+1).getLastUpdatedDate())){
				fail("Appointment Requests are not ordered by last updated date");
			}
		}
	}
	
	@Test
	public void fetchAppointmentsRequestsUsingFilterAndCheckTheyAreSortedByLastUpdatedDateInAscendingOrder() {
		AppointmentRequests appointmentRequests = dataservice.getAppointmentRequests(null);
		
		assertTrue(appointmentRequests.size()>0);

		for(int i=0; i < appointmentRequests.size()-1; i++){
			if(appointmentRequests.get(i).getLastUpdatedDate().after(appointmentRequests.get(i+1).getLastUpdatedDate())){
				fail("Appointment Requests are not ordered by last updated date");
			}
		}
	}

	@Test
	public void fetchListOfAppointmentRequestsReturnsEmpty() {
		DateFilter dateFilter = DateFilterFactory.createEmptyFilter();
		AppointmentRequests appointmentRequests = dataservice.getPatientAppointmentRequests(patientIdentifierWithNoData, dateFilter, ScopeFilter.getInstanceForLongitudinalScope());
		assertEquals(0, appointmentRequests.size());
	}

	@Test
	public void getAppointmentRequest() {
		AppointmentRequest fetchedAppointmentRequest = dataservice.getPatientAppointmentRequest(patientIdentifier, dataIdentifier);
		assertEquals(persistedAppointmentRequest, fetchedAppointmentRequest);
	}

    @Test
	public void saveAppointmentRequests() {
		AppointmentRequest savedAppointmentRequest = dataservice.saveAppointmentRequest(pilotSites, appointmentRequest, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());
		assertNotNull(savedAppointmentRequest);
		assertNotNull(savedAppointmentRequest.getDataIdentifier());
		assertNotNull(savedAppointmentRequest.getDataIdentifier().getUniqueId());
		
		assertNotNull(savedAppointmentRequest.getPatient());
		assertTrue("Patient FirstName isNullish", NullChecker.isNotNullish(savedAppointmentRequest.getPatient().getFirstName()));
		assertTrue("Patient LastName isNullish", NullChecker.isNotNullish(savedAppointmentRequest.getPatient().getLastName()));
		
		assertEquals(savedAppointmentRequest.isTextMessagingAllowed(), savedAppointmentRequest.getPatient().isTextMessagingAllowed());
		assertEquals(savedAppointmentRequest.getTextMessagingPhoneNumber(), savedAppointmentRequest.getPatient().getTextMessagingPhoneNumber());
		
		dataservice.deleteAppointmentRequest(savedAppointmentRequest, ScopeFilter.getInstanceForLongitudinalScope());
	}

	@Test
	public void validateOnlyOnePrimaryCareAppointmentCanBeSubmitted() {
		boolean didReceiveException = false;
		
		AppointmentRequest newRequest1 = createValidAppointmentRequest();
		AppointmentRequest newRequest2 = createValidAppointmentRequest();
		
		newRequest1.setAppointmentType(AppointmentRequestType.PRIMARY_CARE.getName());
		newRequest2.setAppointmentType(AppointmentRequestType.PRIMARY_CARE.getName());
		
		AppointmentRequest savedAppointmentRequest = null;
		AppointmentRequest savedAppointmentRequest2 = null;
		try{
			savedAppointmentRequest = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());
			savedAppointmentRequest2 = dataservice.saveAppointmentRequest(pilotSites, newRequest2, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());
		}catch(InvalidAppointmentRequestException exception){
			didReceiveException = true;
		}
		
		assertTrue(didReceiveException);
		
		if (savedAppointmentRequest != null) {
			dataservice.deleteAppointmentRequest(savedAppointmentRequest, ScopeFilter.getInstanceForLongitudinalScope());
		}
		if (savedAppointmentRequest2 != null) {
			dataservice.deleteAppointmentRequest(savedAppointmentRequest2, ScopeFilter.getInstanceForLongitudinalScope());
		}
	}

	@Test
	public void validateOnlyTwoMentalHealthAppointmentCanBeSubmitted() {
		boolean didReceiveException = false;

		AppointmentRequest newRequest1 = createValidAppointmentRequest();
		AppointmentRequest newRequest2 = createValidAppointmentRequest();

		AppointmentRequest savedAppointmentRequest1 = null;
		AppointmentRequest savedAppointmentRequest2 = null;
		try{
			savedAppointmentRequest1 = dataservice.saveAppointmentRequest(pilotSites, newRequest1, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());
			savedAppointmentRequest2 = dataservice.saveAppointmentRequest(pilotSites, newRequest2, haMode, true, ScopeFilter.getInstanceForLongitudinalScope());
		}catch(InvalidAppointmentRequestException exception){
			didReceiveException = true;
		}
		
		assertTrue(didReceiveException);
		
		if (savedAppointmentRequest1 != null) {
			dataservice.deleteAppointmentRequest(savedAppointmentRequest1, ScopeFilter.getInstanceForLongitudinalScope());
		}
		if (savedAppointmentRequest2 != null) {
			dataservice.deleteAppointmentRequest(savedAppointmentRequest2, ScopeFilter.getInstanceForLongitudinalScope());
		}
	}
	
	private AppointmentRequest createValidAppointmentRequest() {
		String appointmentType = AppointmentRequestType.MENTAL_HEALTH.getName();
		String email = "test@agilex.com";
		String phoneNumber = "+1 (555) 555-5555 x55.55";
		String visitType = AppointmentRequestVisitType.OFFICE_VISIT.getName();
		String purposeOfVisit = AppointmentRequestPurposeOfVisit.OTHER.getName();
		String otherPurposeOfVisit = "Other purpose of visit specified test";
		String providerId = "PROV1";
		boolean secondRequest = false;
		
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, 7);
		String optionDate1 = DateHelper.formatDate(cal.getTime());
		String optionTime1 = "AM";
		cal.add(Calendar.DATE, 1);
		String optionDate2 = DateHelper.formatDate(cal.getTime());
		String optionTime2 = "PM";
		cal.add(Calendar.DATE, 2);
		String optionDate3 = DateHelper.formatDate(cal.getTime());
		String optionTime3 = "AM";
		HashSet<String> bestTimetoCall = new HashSet<String>();
		bestTimetoCall.add("9 AM - 11 AM");
		bestTimetoCall.add("11 AM - 1 PM");
		String textMessagingPhoneNumber = "555-222-3242";
		boolean textMessagingAllowed = true;
		
		String facilityAddress = "test address";
		String facilityCity = "test city";
		String facilityState = "test state";
		String facilityCode = "537";
		String facilityName = "Jesse Brown VAMC";
		String facilityParentSiteCode = "537";
		String facilityType = "test facility type";
		
		Facility facility = new Facility();
		facility.setAddress(facilityAddress);
		facility.setCity(facilityCity);
		facility.setFacilityCode(facilityCode);
		facility.setName(facilityName);
		facility.setParentSiteCode(facilityParentSiteCode);
		facility.setState(facilityState);
		facility.setType(facilityType);
		
		AppointmentRequest a = new AppointmentRequest();
		
		a.setPatientIdentifier(patientIdentifier);
		a.setAppointmentType(appointmentType);
		a.setVisitType(visitType);
		a.setEmail(email);
		a.setPhoneNumber(phoneNumber);
		a.setOptionDate1(optionDate1);
		a.setOptionTime1(optionTime1);
		a.setOptionDate2(optionDate2);
		a.setOptionTime2(optionTime2);
		a.setOptionDate3(optionDate3);
		a.setOptionTime3(optionTime3);
		a.setFacility(facility);
		a.setBestTimetoCall(bestTimetoCall);
		a.setPurposeOfVisit(purposeOfVisit);
		a.setOtherPurposeOfVisit(otherPurposeOfVisit);
		a.setStatus(AppointmentRequestStatus.SUBMITTED.getName());
		a.setProviderId(providerId);
		a.setLastUpdatedDate(new Date());
		a.setSecondRequest(secondRequest);
		a.setTextMessagingAllowed(textMessagingAllowed);
		a.setTextMessagingPhoneNumber(textMessagingPhoneNumber);
		a.setPatientIdentifier(patientIdentifier);
		a.setCreatedDate(new Date());
		
		return a;
	}
}
