package com.agilex.healthcare.mobilehealthplatform.datalayer.securemessage;

import static org.junit.Assert.*;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class SecureMessageDataLayerMockTest {

	SecureMessageDataLayerMock datalayer = new SecureMessageDataLayerMock();
	String defaultSmIcn = "71";

	@Test
	public void verifyFetchGallowUser() {
		SecureMessageUser secureMessageUser = datalayer.fetchSecureMessageUser("D123401");
		assertNotNull(secureMessageUser);
	}

	@Test
	public void verifyFetchFolders() {
		SecureMessageFolders folders = datalayer.fetchFolders(this.defaultSmIcn);

		assertNotNull(folders);
		assertTrue(folders.size() >= 5);
	}

	@Test
	public void verifyFetchContacts() {
		SecureMessageContacts contacts = datalayer.fetchContacts(this.defaultSmIcn);

		assertNotNull(contacts);
		assertTrue(contacts.size() >= 5);
	}

	@Test
	public void verifyReadMessage() {
		SecureMessages inboxMessages = datalayer.fetchSecureMessages(this.defaultSmIcn, new SecureMessageFilter("0"));
		SecureMessage readme = inboxMessages.get(0);
		int oplock = Integer.parseInt(readme.getAddresseeOplock());
		SecureMessage read = datalayer.readSecureMessage(this.defaultSmIcn, readme);

		assertNotNull(read.getMessageBody());
		assertNotNull(read.getReadDate());
		assertEquals(oplock + 1, Integer.parseInt(read.getAddresseeOplock()));
	}

	@Test(expected = RuntimeException.class)
	public void verifyReadChecksOplockState() { 
		// oplock must be the same (if it's not, someone has modified it since then)
		SecureMessages inboxMessages = datalayer.fetchSecureMessages(this.defaultSmIcn, new SecureMessageFilter("0"));
		SecureMessage readme = inboxMessages.get(0);
		readme.setAddresseeOplock("10");
		datalayer.readSecureMessage(this.defaultSmIcn, readme);
	}

	@Test
	public void verifyChangeSystemFolderNameSetsFields() {
		SecureMessage secureMessage = new SecureMessage();
		secureMessage.setFolderId(SecureMessageCode.FOLDER_ID_INBOX.toString());
		secureMessage.setPatientId(this.defaultSmIcn);

		datalayer.changeFolderName(secureMessage);

		assertEquals(SecureMessageCode.FOLDER_NAME_INBOX, secureMessage.getFolder());
	}

	@Test
	public void verifyChangeCustomFolderNameSetsFields() {
		SecureMessage secureMessage = new SecureMessage();
		secureMessage.setFolderId("3000");
		secureMessage.setPatientId(this.defaultSmIcn);

		datalayer.changeFolderName(secureMessage);

		assertEquals("Pharmacy", secureMessage.getFolder());
	}

	@Test
	public void verifyMoveRemovesFromPreviousFolder() {
		SecureMessages deletedMessages = datalayer.fetchSecureMessages(this.defaultSmIcn, new SecureMessageFilter(SecureMessageCode.FOLDER_ID_DELETED.toString()));
		SecureMessage messageToMoveFromDeleted = deletedMessages.get(0);
		datalayer.moveSecureMessage(this.defaultSmIcn, messageToMoveFromDeleted.getUniqueId(), SecureMessageCode.FOLDER_ID_INBOX.toString());

		assertTrue(validateInFolder(this.defaultSmIcn, SecureMessageCode.FOLDER_ID_INBOX.toString(), messageToMoveFromDeleted.getUniqueId()));
		assertTrue(validateNotInFolder(this.defaultSmIcn, SecureMessageCode.FOLDER_ID_DELETED.toString(), messageToMoveFromDeleted.getUniqueId()));
	}

	@Test
	public void verifyClearBody() {
		SecureMessage message = new SecureMessage();
		message.setMessageBody("body should be set to null");

		SecureMessages secureMessages = new SecureMessages();
		secureMessages.add(message);

		SecureMessages clearedBodies = datalayer.clearBodies(secureMessages);

		SecureMessage message0 = clearedBodies.get(0);
		assertEquals("", message0.getMessageBody());
	}

	@Test(expected = RuntimeException.class)
	public void verifyMoveToSendFails() {
		SecureMessages inboxMessages = datalayer.fetchSecureMessages(this.defaultSmIcn, new SecureMessageFilter("0"));
		SecureMessage inboxToMove = inboxMessages.get(0);

		datalayer.moveSecureMessage(this.defaultSmIcn, inboxToMove.getUniqueId(), SecureMessageCode.FOLDER_ID_SENT.toString());
	}

	@Test(expected = RuntimeException.class)
	public void verifyMoveToDraftFails() {
		SecureMessages inboxMessages = datalayer.fetchSecureMessages(this.defaultSmIcn, new SecureMessageFilter("0"));
		SecureMessage inboxToMove = inboxMessages.get(0);

		datalayer.moveSecureMessage(this.defaultSmIcn, inboxToMove.getUniqueId(), SecureMessageCode.FOLDER_ID_DRAFT.toString());
	}

	@Test(expected = RuntimeException.class)
	public void verifyMoveWithoutFolderFails() {
		SecureMessages inboxMessages = datalayer.fetchSecureMessages(this.defaultSmIcn, new SecureMessageFilter("0"));
		SecureMessage inboxToMove = inboxMessages.get(0);

		datalayer.moveSecureMessage(this.defaultSmIcn, inboxToMove.getUniqueId(), null);
	}

	@Test
	public void verifySaveDraft() {
		SecureMessage savedDraft = createDraft();

		assertTrue(validateInFolder(this.defaultSmIcn, SecureMessageCode.FOLDER_ID_DRAFT.toString(), savedDraft.getUniqueId()));
	}
	
	@Test
	public void validateDraftSortOrder() {
		SecureMessageFilter filter = new SecureMessageFilter(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		SecureMessages secureMessages = datalayer.fetchSecureMessages(this.defaultSmIcn, filter);
		
		for(int i = 0; i < secureMessages.size() - 1; i++) {
			String add1 = secureMessages.get(i).getAddresseeId();
			String add2 = secureMessages.get(i + 1).getAddresseeId();
			
			assertTrue(Integer.parseInt(add1) > Integer.parseInt(add2));
		}
	}

	@Test
	public void verifySaveDraftTwiceShouldNotDuplicate() {
		SecureMessage savedDraft = createDraft();
		savedDraft.setMessageBody("2.0");
		SecureMessage savedDraft2 = datalayer.saveSecureMessageDraft(this.defaultSmIcn, savedDraft);

		assertEquals("2.0", savedDraft2.getMessageBody());
		assertEquals(savedDraft.getUniqueId(), savedDraft2.getUniqueId());

		int count = 0;
		SecureMessages drafts = datalayer.fetchSecureMessages(this.defaultSmIcn, new SecureMessageFilter(SecureMessageCode.FOLDER_ID_DRAFT.toString(), "1", "1000"));
		for (SecureMessage draft : drafts) {
			if (draft.getUniqueId().equals(savedDraft.getUniqueId())) {
				count++;
			}
		}

		assertEquals(1, count);
	}
	
	@Test
	public void verifySendDraft() {
		SecureMessage draft = get0thMessageFromFolder(this.defaultSmIcn, SecureMessageCode.FOLDER_ID_DRAFT.toString());
		SecureMessage sentDraft = datalayer.sendSecureMessageDraft(this.defaultSmIcn, draft);
		
		assertNotNull(sentDraft.getDateSent());
		
		assertTrue(validateInFolder(this.defaultSmIcn, SecureMessageCode.FOLDER_ID_SENT.toString(), draft.getUniqueId()));
		assertTrue(validateNotInFolder(this.defaultSmIcn, SecureMessageCode.FOLDER_ID_DRAFT.toString(), draft.getUniqueId()));
	}

	@Test
	public void verifyDeleteHelper() {
		SecureMessages inboxMessages = datalayer.fetchSecureMessages(this.defaultSmIcn, new SecureMessageFilter("0"));
		SecureMessage messageToDelete = inboxMessages.get(0);

		datalayer.deleteSecureMessage(this.defaultSmIcn, messageToDelete.getUniqueId());

		assertTrue(validateNotInFolder(this.defaultSmIcn, SecureMessageCode.FOLDER_ID_INBOX.toString(), messageToDelete.getUniqueId()));
	}

	public void verifySetMessageIds() {
		SecureMessage secureMessage = new SecureMessage();
		datalayer.setMessageIds(secureMessage);

		assertNotNull(secureMessage.getAddresseeId());
		assertNotNull(secureMessage.getAddresseeOplock());
		assertNotNull(secureMessage.getThreadId());
		assertNotNull(secureMessage.getThreadOplock());
		assertNotNull(secureMessage.getUniqueId());
		assertNotNull(secureMessage.getMessageOplock());
	}

	public SecureMessage createDraft() {
		SecureMessage secureMessageDraft = new SecureMessage();
		secureMessageDraft.setMessageBody("body of draft");
		secureMessageDraft.setSubject("draft subject");
		secureMessageDraft.setFrom("Me");
		secureMessageDraft.setFromId("D123401");
		secureMessageDraft.setToId("9876");
		secureMessageDraft.setTo("My Friend");
		secureMessageDraft.setPatientIdentifier("mock", this.defaultSmIcn);

		return datalayer.saveSecureMessageDraft(this.defaultSmIcn, secureMessageDraft);
	}
	
	public SecureMessage get0thMessageFromFolder(String userId, String folderId) {
		SecureMessages secureMessages = datalayer.fetchSecureMessages(userId, new SecureMessageFilter(folderId));
		return secureMessages.get(0);
	}

	boolean validateInFolder(String userId, String folderId, String messageId) {
		SecureMessageFilter filter = new SecureMessageFilter(folderId, "1", "1000");
		SecureMessages messagesInFolder = datalayer.fetchSecureMessages(userId, filter);
		for (SecureMessage message : messagesInFolder) {
			if (NullSafeStringComparer.areEqual(message.getUniqueId(), messageId)) {
				return true;
			}
		}
		return false;
	}

	boolean validateNotInFolder(String userId, String folderId, String messageId) {
		return !validateInFolder(userId, folderId, messageId);
	}

	@Test(expected = RuntimeException.class)
	public void verifyMissingReplyToIdFails() {
		SecureMessage message = new SecureMessage();
		datalayer.replySecureMessage(this.defaultSmIcn, message);
	}
}
