package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;
import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.Notification;
import com.agilex.healthcare.mobilehealthplatform.domain.Notifications;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.utility.DateHelper;

public class NotificationDataLayerExcelTest {
	
	NotificationDataLayer dl = new NotificationDataLayerExcel();
	DateFilter emptyFilter = DateFilterFactory.createEmptyFilter();

	@Test
	public void getPrestagedNotifications() {
		String userId = "PATID26";
		NotificationDataLayer dl = new NotificationDataLayerExcel();
		Notifications notifications = dl.getActiveNotifications(userId, emptyFilter);
		assertNotNull(notifications);
		assertEquals(3, notifications.size());
		assertTrue(areAllNotificationsActive(notifications));

		Notification retrievedNotification = notifications.get(0);
		Notification specificNotification = dl.getNotification(userId, retrievedNotification.getNotificationId());
		assertNotNull(specificNotification);
		
		assertEquals(retrievedNotification.getUserId(), specificNotification.getUserId());
		assertEquals(retrievedNotification.getNotificationId(), specificNotification.getNotificationId());
		assertEquals(retrievedNotification.getDate(), specificNotification.getDate());
		assertEquals(retrievedNotification.getBody(), specificNotification.getBody());
	}
	
	@Test
	public void get0ResultsForInvalidUser() {
		String patientId = "invalid";
		Notifications notifications = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(0, notifications.size());
	}
	
	@Test
	public void addAndRetrieveNotifications() {
		String patientId = generateId();
		Notifications notifications = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(0, notifications.size());
		
		Notification notification = new Notification();
		notification.setPatientId(patientId);
		Notification savedEx = dl.saveNotification(notification);
		
		Notifications retrieved = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(1, retrieved.size());
		assertEquals(savedEx.getUniqueId(), retrieved.get(0).getUniqueId());
	}
	
	@Test
	public void addAndRetrieveSortedNotifications() {
		String patientId = generateId();
		String testBody = "TEST BODY";

		Notifications notifications = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(0, notifications.size());
		
		Notification notification = new Notification();
		notification.setPatientId(patientId);
		notification.setBody(testBody);
		notification.setActiveFlag(true);
		notification.setDate(new Date());
		dl.saveNotification(notification);
		
		Notification notification2 = new Notification();
		notification2.setPatientId(patientId);
		notification2.setBody(testBody);
		notification2.setActiveFlag(true);
		notification2.setDate(new Date());
		Notification savedEx2 = dl.saveNotification(notification2);
		
		Notifications retrieved = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(2, retrieved.size());
		assertEquals(savedEx2.getUniqueId(), retrieved.get(0).getUniqueId());
	}
	
	@Test
	public void addAndRetrieve50Notifications() {
		String patientId = generateId();
		String testBody = "TEST BODY";

		Notifications notifications = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(0, notifications.size());
		
		for (int i=0; i<100; i++) {
			Notification notification = new Notification();
			notification.setPatientId(patientId);
			notification.setBody(testBody);
			notification.setActiveFlag(true);
			notification.setDate(DateHelper.parseDate("1/12/2012"));
			dl.saveNotification(notification);
		}
		
		Notifications retrieved = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(50, retrieved.size());
	}

	@Test 
	public void validateUpdate() {
		String patientId = generateId();
		String testBody = "TEST BODY";
		String testBody2 = "TEST BODY2";
		
		Notification notification = new Notification();
		notification.setPatientId(patientId);
		notification.setBody(testBody);
		notification.setActiveFlag(true);
		notification.setDate(DateHelper.parseDate("1/12/2012"));
		Notification saved = dl.saveNotification(notification);
		assertNotNull(saved);
		assertEquals(testBody, saved.getBody());
		
		saved.setBody(testBody2);
		Notification updated = dl.saveNotification(saved);
		
		assertEquals(testBody2, updated.getBody());
	}
	
	
	@Test 
	public void validateTodaysNotifications() {
		String patientId = generateId();
		String testBody = "TEST BODY";

		Notifications notifications = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(0, notifications.size());
		
		Notification notification = new Notification();
		notification.setPatientId(patientId);
		notification.setBody(testBody);
		notification.setActiveFlag(true);
		notification.setDate(new Date());
		dl.saveNotification(notification);
		
		Notification notification2 = new Notification();
		notification2.setPatientId(patientId);
		notification2.setBody(testBody);
		notification2.setActiveFlag(true);
		notification2.setDate(DateHelper.parseDate("1/1/2012"));
		dl.saveNotification(notification2);
		
		Notifications allNotifications = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(2, allNotifications.size());
		
		DateFilter todayFilter = DateFilterFactory.createFilterFromToday();
		Notifications todaysNotifications = dl.getActiveNotifications(patientId, todayFilter);
		assertEquals(1, todaysNotifications.size());
	}
	
	@Test 
	public void validateDelete() {
		String patientId = generateId();
		
		Notification notification = new Notification();
		notification.setPatientId(patientId);
		notification.setDate(DateHelper.parseDate("1/12/2012"));
		dl.saveNotification(notification);
		
		Notifications shouldBeOne = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(1, shouldBeOne.size());
		
		dl.deleteNotification(shouldBeOne.get(0));
		
		Notifications shouldBeEmpty = dl.getActiveNotifications(patientId, emptyFilter);
		assertEquals(0, shouldBeEmpty.size());
	}
	
	private String generateId() {
		return UUID.randomUUID().toString();
	}

	private boolean areAllNotificationsActive(Notifications notifications) {
		boolean result = true;
		for (Notification notification : notifications) {
			if (notification.isActiveFlag() == false) {
				result = false;
				break;
			}
		}
		return result;
	}
}
