package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistrations;

public class DeviceRegistrationDataLayerMockTest {

	DeviceRegistrationDataLayer dl = new DeviceRegistrationDataLayerMock();

	private void init(String userId) {
		DeviceRegistration deviceRegistration = createDeviceRegistration(userId, "testtokenfor" + userId);
		dl.saveRegistration(deviceRegistration);

		deviceRegistration = createDeviceRegistration(userId, "anothertesttokenfor" + userId);
		dl.saveRegistration(deviceRegistration);
	}

	private void printDeviceRegistrations(DeviceRegistrations deviceRegistrations) {
		for (DeviceRegistration deviceRegistration : deviceRegistrations) {
			System.out.println(String.format("Device Registration user [%s] token [%s]", deviceRegistration.getUserId(), deviceRegistration.getDeviceToken()));
		}
	}

	@Test
	public void verifygetAllDeviceRegistrationsById() {
		String userId = "01";
		init(userId);
		DeviceRegistrations deviceRegistrations = dl.getRegistrations(userId);
		printDeviceRegistrations(deviceRegistrations);
		assertEquals(2, deviceRegistrations.size());
	}

	@Test
	public void verfiyGetByDeviceRegistrationId() {
		String userId = "02";
		DeviceRegistration savedDeviceRegistration = createDeviceRegistration(userId, "testtokenfor" + userId);
		savedDeviceRegistration = dl.saveRegistration(savedDeviceRegistration);
		assertNotNull(savedDeviceRegistration);
		assertNotNull(savedDeviceRegistration.getDeviceRegistrationId());
		assertEquals(userId, savedDeviceRegistration.getUserId());

		DeviceRegistration retrievedDeviceRegistration = dl.getRegistration(userId, savedDeviceRegistration.getDeviceToken());
		checkDeviceRegistrationsAreSame(savedDeviceRegistration, retrievedDeviceRegistration);
	}

	@Test
	public void verifySaveDeviceRegistrationWithExistingPatientId() {
		String userId = "03";
		DeviceRegistration deviceRegistrationToSave = createDeviceRegistration(userId, "testtokenfor" + userId);
		DeviceRegistrationDataLayer dl = new DeviceRegistrationDataLayerMock();
		DeviceRegistration savedDeviceRegistration = dl.saveRegistration(deviceRegistrationToSave);

		assertNotNull(savedDeviceRegistration);
		assertNotNull(savedDeviceRegistration.getDeviceRegistrationId());
		checkDeviceRegistrationsAreSame(deviceRegistrationToSave, savedDeviceRegistration);
	}

	@Test
	public void verifySaveDeviceRegistrationWithNewPatientId() {
		String userId = "uniqueId";

		DeviceRegistration deviceRegistrationToSave = createDeviceRegistration(userId, "testtokenfor" + userId);

		DeviceRegistrationDataLayer dl = new DeviceRegistrationDataLayerMock();
		DeviceRegistration savedDeviceRegistration = dl.saveRegistration(deviceRegistrationToSave);

		assertNotNull(savedDeviceRegistration.getDeviceRegistrationId());
		assertEquals(userId, deviceRegistrationToSave.getUserId());

		DeviceRegistration retrieved = dl.getRegistration(userId, savedDeviceRegistration.getDeviceToken());
		assertEquals(userId, retrieved.getUserId());

		assertNotNull(retrieved.getDeviceRegistrationId());
		assertEquals(deviceRegistrationToSave.getDeviceToken(), savedDeviceRegistration.getDeviceToken());
		assertEquals(deviceRegistrationToSave.getRegisteredDate(), savedDeviceRegistration.getRegisteredDate());
	}

	@Test
	public void verifyDeleteDeviceRegistration() {
		String userId = "08";
		DeviceRegistration deviceRegistration = createDeviceRegistration(userId, "testtokenfor" + userId);
		DeviceRegistration savedDeviceRegistration = dl.saveRegistration(deviceRegistration);

		assertNotNull(savedDeviceRegistration.getDeviceRegistrationId());

		DeviceRegistration deviceRegistrationToDelete = new DeviceRegistration();
		deviceRegistrationToDelete.setUserId(savedDeviceRegistration.getUserId());
		deviceRegistrationToDelete.setDeviceToken(savedDeviceRegistration.getDeviceToken());

		dl.deleteRegistration(deviceRegistrationToDelete);

		DeviceRegistration registration = dl.getRegistration(userId, savedDeviceRegistration.getDeviceToken());
		assertNull(registration);
	}

	@Test
	public void verifyFetchOfDeviceRegistrationsForMultipleUsers() {
		String userId1 = "09";
		String userId2 = "10";

		List<String> userIds = new ArrayList<String>();
		userIds.add(userId1);
		userIds.add(userId2);

		init(userId1);
		init(userId2);

		DeviceRegistrations fetchedDeviceRegistrations = dl.fetchDeviceRegistrations(userIds);

		assertEquals(fetchedDeviceRegistrations.size(), 4);

		for (DeviceRegistration fetchedDeviceRegistration : fetchedDeviceRegistrations) {
			String token = fetchedDeviceRegistration.getDeviceToken();
			String userId = fetchedDeviceRegistration.getUserId();

			boolean userId1Good = userId.equals(userId1) && (token.equals("testtokenfor" + userId1) || token.equals("anothertesttokenfor" + userId1));
			boolean userId2Good = userId.equals(userId2) && (token.equals("testtokenfor" + userId2) || token.equals("anothertesttokenfor" + userId2));

			assertTrue(userId1Good || userId2Good);
		}
	}
	
	@Test
	public void verifyOptedOutRegistrationNotReturned() {
		DeviceRegistrationDataLayer dl = new DeviceRegistrationDataLayerMock();
		String userId = UUID.randomUUID().toString();
		String deviceToken = UUID.randomUUID().toString();
		
		DeviceRegistration deviceRegistration = createDeviceRegistration(userId, deviceToken);
		deviceRegistration.setOptedIn(false);
		dl.saveRegistration(deviceRegistration);
		
		DeviceRegistrations retrievedRegistrations = dl.getRegistrations(userId);
		assertEquals(0, retrievedRegistrations.size());
		
		DeviceRegistration retrievedRegistration = dl.getRegistration(userId, deviceToken);
		assertNotNull(retrievedRegistration);
		assertEquals(false, retrievedRegistration.isOptedIn());
	}

	private DeviceRegistration createDeviceRegistration(String userId, String deviceToken) {
		DeviceRegistration deviceRegistration = new DeviceRegistration();
		deviceRegistration = new DeviceRegistration();
		deviceRegistration.setUserId(userId);
		deviceRegistration.setFirstName(userId + "first");
		deviceRegistration.setLastName(userId + "last");
		deviceRegistration.setDeviceToken(deviceToken);
		deviceRegistration.setRegisteredDate(new Date());
		deviceRegistration.setOptedIn(true);
		return deviceRegistration;
	}

	private void checkDeviceRegistrationsAreSame(DeviceRegistration expected, DeviceRegistration actual) {
		assertEquals(expected.getDeviceRegistrationId(), actual.getDeviceRegistrationId());
		assertEquals(expected.getDeviceToken(), actual.getDeviceToken());
		assertEquals(expected.getRegisteredDate(), actual.getRegisteredDate());
		assertEquals(expected.getFirstName(), actual.getFirstName());
		assertEquals(expected.getLastName(), actual.getLastName());
		assertEquals(expected.getUserId(), actual.getUserId());
	}

}
