package com.agilex.healthcare.mobilehealthplatform.datalayer.mood;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.event.MoodEventDataLayerMock;
import com.agilex.healthcare.mobilehealthplatform.domain.MoodEvent;
import com.agilex.healthcare.mobilehealthplatform.domain.MoodEvents;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.utility.DateHelper;

public class MoodEventDataLayerMockTest {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MoodEventDataLayerMockTest.class);

	MoodEventDataLayerMock dl = new MoodEventDataLayerMock();

	@Test
	public void testGallowMoodsExist() {
		String gallowPatientId = "D123401";
		MoodEvents events = dl.getMoodEvents(gallowPatientId);

		assertTrue(events.size() >= 5);
	}

	@Test
	public void testSaveAndRetrieve() {
		String patientId = "01";
		MoodEvent mood = createMoodEvent(patientId, "Wonderful");
		MoodEvent savedMood = dl.saveMoodEvent(mood);

		MoodEvents moodEvents = dl.getMoodEvents(patientId);
		assertEquals(1, moodEvents.size());
		assertEquals("Wonderful", savedMood.getValue());
		assertEquals("Wonderful01", savedMood.getNotes());

		MoodEvent retrievedEvent = dl.getMoodEvent(patientId, savedMood.getUniqueId());
		assertNotNull(retrievedEvent);
		assertEquals(savedMood.getUniqueId(), retrievedEvent.getUniqueId());
		assertEquals("Wonderful", retrievedEvent.getValue());
	}

	@Test
	public void testDelete() {
		String patientId = "02";
		MoodEvent mood = createMoodEvent(patientId, "Delete");
		MoodEvent savedEvent = dl.saveMoodEvent(mood);
		assertEquals("Delete", savedEvent.getValue());
		assertNotNull(savedEvent.getUniqueId());

		dl.deleteMoodEvent(savedEvent);

		MoodEvent retrievedEvent = dl.getMoodEvent(patientId, savedEvent.getUniqueId());
		assertNull(retrievedEvent);
	}

	@Test
	public void testFilteringGet() {
		String patientId = "03";
		MoodEvent mood01 = createMoodEvent(patientId, "Filter 01");
		mood01.setDate(DateHelper.parseDate("1/1/2012"));
		dl.saveMoodEvent(mood01);

		MoodEvent mood02 = createMoodEvent(patientId, "Filter 02");
		mood02.setDate(DateHelper.parseDate("1/1/2013"));
		dl.saveMoodEvent(mood02);

		MoodEvents retrieved = dl.getMoodEvents(patientId);
		assertEquals(2, retrieved.size());

		DateFilter filter = DateFilterFactory.createFilterFromDate("1/1/2012", "6/6/2012");

		retrieved = dl.getMoodEvents(patientId, filter);
		assertEquals(1, retrieved.size());
		MoodEvent event = retrieved.get(0);
		assertEquals("Filter 01", event.getValue());
	}

	@Test
	public void verifyExistingDataAfterMockReInit() {
		String patientId = UUID.randomUUID().toString();

		logger.debug(String.format("==== Reinit [patientId:%s]", patientId));

		MoodEvent event = createMoodEvent(patientId, "01");
		dl.saveMoodEvent(event);
		MoodEvents retrieved = dl.getMoodEvents(patientId);
		assertEquals(1, retrieved.size());

		dl = new MoodEventDataLayerMock();
		MoodEvents retrievedFromNewMock = dl.getMoodEvents(patientId);
		assertEquals(1, retrievedFromNewMock.size());
	}

	private MoodEvent createMoodEvent(String patientId, String mood) {

		MoodEvent event = new MoodEvent();

		event.setPatientId(patientId);
		event.setDate(DateHelper.parseDate("1/1/2012"));
		event.setValue(mood);
		event.setNotes(mood + patientId);

		return event;
	}

	@Test
	public void filterByDate() {
		MoodEventDataLayer dao = new MoodEventDataLayerMock();
		DateFilter filter = DateFilterFactory.createEmptyFilter();
		MoodEvents events = dao.getMoodEvents("D123401", filter);
		assertEquals(5, events.size());

		dao = new MoodEventDataLayerMock();
		filter = DateFilterFactory.createFilterFromDate("01/10/2012");
		events = dao.getMoodEvents("D123401", filter);
		assertEquals(1, events.size());

		dao = new MoodEventDataLayerMock();
		filter = DateFilterFactory.createEmptyFilter();
		events = dao.getMoodEvents("D123401", filter);
		assertEquals(5, events.size());
	}
}
