package com.agilex.healthcare.mobilehealthplatform.datalayer.mood;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;
import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.event.MoodEventDataLayerExcel;
import com.agilex.healthcare.mobilehealthplatform.domain.MoodEvent;
import com.agilex.healthcare.mobilehealthplatform.domain.MoodEvents;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.utility.DateHelper;

public class MoodEventDataLayerExcelTest {

	MoodEventDataLayer dl = new MoodEventDataLayerExcel();
	private static final String patientIdGallow = "D123401";

	@Test
	public void getPrestagedMoodEvents() {
		MoodEvents moodEvents = dl.getMoodEvents(patientIdGallow);
		assertTrue(moodEvents.size() > 0);

		MoodEvent moodEvent01 = moodEvents.get(0);
		assertEquals("1", moodEvent01.getUniqueId());

		assertEquals(DateHelper.parseDate("1/1/2012"), moodEvent01.getDate());
	}

	@Test
	public void get0ResultsForInvalidPatient() {
		String patientId = "invalid";
		MoodEvents moodEvents = dl.getMoodEvents(patientId);
		assertEquals(0, moodEvents.size());
	}

	@Test
	public void addAndRetrieveMoodEvents() {
		String patientId = generateId();
		MoodEvents moodEvents = dl.getMoodEvents(patientId);
		assertEquals(0, moodEvents.size());

		MoodEvent moodEvent = new MoodEvent();
		moodEvent.setPatientId(patientId);
		MoodEvent savedMoodEvent = dl.saveMoodEvent(moodEvent);

		MoodEvents retrieved = dl.getMoodEvents(patientId);
		assertEquals(1, retrieved.size());
		assertEquals(savedMoodEvent.getUniqueId(), retrieved.get(0).getUniqueId());
	}

	@Test
	public void retrieveWithDateFilter() {
		String patientId = generateId();

		MoodEvent moodEvent = new MoodEvent();
		moodEvent.setPatientId(patientId);
		moodEvent.setDate(DateHelper.parseDate("1/12/2012"));
		dl.saveMoodEvent(moodEvent);

		moodEvent = new MoodEvent();
		moodEvent.setPatientId(patientId);
		moodEvent.setDate(DateHelper.parseDate("1/13/2012"));
		dl.saveMoodEvent(moodEvent);

		moodEvent = new MoodEvent();
		moodEvent.setPatientId(patientId);
		moodEvent.setDate(DateHelper.parseDate("1/15/2012"));
		dl.saveMoodEvent(moodEvent);

		Date startDate = DateHelper.parseDate("1/12/2012");
		Date endDate = DateHelper.parseDate("1/12/2012");
		DateFilter filter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		MoodEvents shouldBeOne = dl.getMoodEvents(patientId, filter);
		assertEquals(1, shouldBeOne.size());		

		startDate = DateHelper.parseDate("1/12/2012");
		endDate = DateHelper.parseDate("1/14/2012");
		filter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		MoodEvents shouldBeTwo = dl.getMoodEvents(patientId, filter);
		assertEquals(2, shouldBeTwo.size());
		
		startDate = DateHelper.parseDate("1/12/2012");
		endDate = DateHelper.parseDate("1/15/2012");
		filter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		MoodEvents shouldBeThree = dl.getMoodEvents(patientId, filter);
		assertEquals(3, shouldBeThree.size());
	}

	@Test
	public void validateUpdate() {
		String patientId = generateId();

		MoodEvent moodEvent = new MoodEvent();
		moodEvent.setPatientId(patientId);
		moodEvent.setValue("1");
		moodEvent.setDate(DateHelper.parseDate("1/12/2012"));
		MoodEvent saved = dl.saveMoodEvent(moodEvent);
		assertNotNull(saved);
		assertEquals("1", saved.getValue());

		saved.setValue("2");
		MoodEvent updated = dl.saveMoodEvent(saved);

		assertEquals("2", updated.getValue());
	}

	@Test
	public void validateDelete() {
		String patientId = generateId();

		MoodEvent moodEvent = new MoodEvent();
		moodEvent.setPatientId(patientId);
		moodEvent.setDate(DateHelper.parseDate("1/12/2012"));
		dl.saveMoodEvent(moodEvent);

		MoodEvents shouldBeOne = dl.getMoodEvents(patientId);
		assertEquals(1, shouldBeOne.size());

		dl.deleteMoodEvent(shouldBeOne.get(0));

		MoodEvents shouldBeEmpty = dl.getMoodEvents(patientId);
		assertEquals(0, shouldBeEmpty.size());
	}

	private String generateId() {
		return UUID.randomUUID().toString();
	}
	
	@Test
	public void testUniqueId() {
		MoodEventDataLayer dl = new MoodEventDataLayerExcel();
		MoodEvents moodEvents = dl.getMoodEvents(patientIdGallow);
		assertTrue(moodEvents.size() > 0);

		MoodEvent moodEvent01 = moodEvents.get(0);
		assertEquals("1", moodEvent01.getUniqueId().toString());

		MoodEvent moodEvent02 = moodEvents.get(1);
		assertEquals("2", moodEvent02.getUniqueId().toString());
	}

	@Test
	public void testDate() {
		MoodEventDataLayer dl = new MoodEventDataLayerExcel();
		MoodEvents moodEvents = dl.getMoodEvents(patientIdGallow);
		assertTrue(moodEvents.size() > 0);

		MoodEvent moodEvent01 = moodEvents.get(0);
		assertEquals(DateHelper.parseDate("1/1/2012"), moodEvent01.getDate());

		MoodEvent moodEvent02 = moodEvents.get(1);
		assertEquals(DateHelper.parseDate("1/2/2012"), moodEvent02.getDate());
	}

	@Test
	public void testValue() {
		MoodEventDataLayer dl = new MoodEventDataLayerExcel();
		MoodEvents moodEvents = dl.getMoodEvents(patientIdGallow);
		assertTrue(moodEvents.size() > 0);

		MoodEvent moodEvent01 = moodEvents.get(0);
		assertEquals("1", moodEvent01.getValue().toString());

		MoodEvent moodEvent02 = moodEvents.get(1);
		assertEquals("2", moodEvent02.getValue().toString());
	}

	@Test
	public void testNote() {
		MoodEventDataLayer dl = new MoodEventDataLayerExcel();
		MoodEvents moodEvents = dl.getMoodEvents(patientIdGallow);
		assertTrue(moodEvents.size() > 0);

		MoodEvent moodEvent01 = moodEvents.get(0);
		assertEquals("note1", moodEvent01.getNotes());
	}
}
