package com.agilex.healthcare.mobilehealthplatform.datalayer.medication;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;

public class MedicationDataLayerMockTest {

	private MedicationDataLayerMock dataLayer;

	@Before
	public void setup() {
		dataLayer = new MedicationDataLayerMock();
	}

	@Test
	public void validateMedicationsAreReturnedForYounger() {
		String patientId = "D123401";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medications medications = dataLayer.getMedications(pId, true);
		assertTrue(medications.size() == 0);
	}

	@Test
	public void validateZeroMedicationsAreReturnedForXYZ() {
		String patientId = "XYZ";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medications medications = dataLayer.getMedications(pId, true);
		assertEquals(0, medications.size());
	}

	@Test
	public void validateZeroMedicationsAreReturnedForNull() {
		String patientId = null;
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medications medications = dataLayer.getMedications(pId, true);
		assertEquals(0, medications.size());
	}

	@Test
	public void validateZeroMedicationsAreReturnedForEmpty() {
		String patientId = null;
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medications medications = dataLayer.getMedications(pId, true);
		assertEquals(0, medications.size());
	}

	@Test
	public void validateMedicationDataIsReturned() {
		String patientId = "D123401";
		String orderNumber = "1";
		String expectedDrugName = "AMITRIPTYLINE HCL 50 MG TAB";
		String expectedMedSoruce = "OUTPATIENT";
		String expectedMedCategory = "Rx";
		String expectedMedPharmacy = "CVS";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medication retrievedMedication = dataLayer.getMedicationLite(pId, orderNumber);
		assertEquals(expectedDrugName, retrievedMedication.getDrugName());
		assertEquals(expectedMedSoruce, retrievedMedication.getMedicationSource());
		assertEquals(expectedMedCategory, retrievedMedication.getCategory());
		assertEquals(expectedMedPharmacy, retrievedMedication.getPharmacy());
		assertNull(retrievedMedication.getMedicationDetail());
	}

	@Test
	public void validateMedicationDataIsNullForInvalidOrderNumber() {
		String patientId = "D123401";
		String orderNumber = "saf";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medication retrievedMedication = dataLayer.getMedicationLite(pId, orderNumber);
		assertNull(retrievedMedication);
	}

	@Test
	public void validateMedicationDataIsNullForInvalidPatientAndOrderNumber() {
		String patientId = "D123401asf";
		String orderNumber = "saf";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medication retrievedMedication = dataLayer.getMedicationLite(pId, orderNumber);
		assertNull(retrievedMedication);
	}

	@Test
	public void validateMedicationDetailIsReturned() {
		String patientId = "D123401";
		String orderNumber = "1";
		String expectedDrugName = "AMITRIPTYLINE HCL 50 MG TAB";
		String expectedMedSoruce = "OUTPATIENT";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medication retrievedMedication = dataLayer.getMedicationDetail(pId, orderNumber);
		assertEquals(expectedDrugName, retrievedMedication.getDrugName());
		assertEquals(expectedMedSoruce, retrievedMedication.getMedicationSource());
		assertTrue(retrievedMedication.getMedicationDetail().contains(expectedDrugName));
	}

	@Test
	public void validateMedicationDetailIsNullForInvalidOrderNumber() {
		String patientId = "D123401";
		String orderNumber = "saf";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medication retrievedMedication = dataLayer.getMedicationDetail(pId, orderNumber);
		assertNull(retrievedMedication);
	}

	@Test
	public void validateMedicationDetailIsNullForInvalidPatientAndOrderNumber() {
		String patientId = "D123401asf";
		String orderNumber = "saf";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medication retrievedMedication = dataLayer.getMedicationDetail(pId, orderNumber);
		assertNull(retrievedMedication);
	}

}
