package com.agilex.healthcare.mobilehealthplatform.datalayer.medication;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.utility.DateHelper;

public class MedicationDataLayerExcelTest {

	private MedicationDataLayer dataLayer;
	private String PATIENTID = "D123401";
	private String ORDERNUM = "1";
	
	private PatientIdentifier createPid(){
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(PATIENTID);
		return pId;
	}
	
	@Before
	public void setup() {
		dataLayer = new MedicationDataLayerExcel();
	}

	@Test
	public void validateMedicationsAreReturnedForYounger() {
		Medications medications = dataLayer.getMedications(createPid(), true);
		assertTrue(medications.size() > 0);
	}

	@Test
	public void validateActive() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		assertEquals(true, medication.isActive());
	}

	@Test
	public void validateDrugName() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		assertEquals("AMITRIPTYLINE HCL 50 MG TAB", medication.getDrugName());
	}

	@Test
	public void validatePharmacy() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		assertEquals("CVS", medication.getPharmacy());
	}

	@Test
	public void validateCategory() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		assertEquals("Rx", medication.getCategory());
	}

	@Test
	public void validateStartDate() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		Date actual = medication.getStartDate();
		Date expected = DateHelper.parseDate("1/1/2012");
		assertEquals(expected, actual);
	}

	@Test
	public void validateEndDate() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		Date d = medication.getEndDate();
		assertEquals(null, d);
	}

	@Test
	public void validateMedSource() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		assertEquals("OUTPATIENT", medication.getMedicationSource());
	}

	@Test
	public void validateSourceSys() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		assertEquals("VA", medication.getSourceSystem());
	}

	@Test
	public void validateDetail() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		assertEquals(null, medication.getMedicationDetail());
	}

	@Test
	public void validatePrescriptionID() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		assertEquals("", medication.getPrescriptionId());
	}

	@Test
	public void validateStatus() {
		Medication medication = dataLayer.getMedicationLite(createPid(), "3");
		assertEquals("ACTIVE", medication.getStatus());
	}

	@Test
	public void validateSig() {
		Medication medication = dataLayer.getMedicationLite(createPid(), "3");
		assertEquals("TAKE ONE TABLET BY MOUTH TWICE A DAY", medication.getSig());
	}

	@Test
	public void validateLastFilledDate() {
		Medication medication = dataLayer.getMedicationLite(createPid(), "3");
		assertEquals(DateHelper.parseDate("1/30/2011"), medication.getLastFilledDate());
	}

	@Test
	public void validateExpiredDate() {
		Medication medication = dataLayer.getMedicationLite(createPid(), ORDERNUM);
		assertEquals(null, medication.getExpiredDate());
	}

	@Test
	public void validateRefills() {
		Medication medication = dataLayer.getMedicationLite(createPid(), "3");
		assertEquals("3", medication.getRefills());
	}

	@Test
	public void validateQuantity() {
		Medication medication = dataLayer.getMedicationLite(createPid(), "3");
		assertEquals("180", medication.getQuantity());
	}

	@Test
	public void validateDaysSupply() {
		Medication medication = dataLayer.getMedicationLite(createPid(), "3");
		assertEquals("90", medication.getDaysSupply());
	}
	
	@Test
	public void validateZeroMedicationsAreReturnedForXYZ() {
		String patientId = "XYZ";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medications medications = dataLayer.getMedications(pId, true);
		assertEquals(0, medications.size());
	}

	@Test
	public void validateZeroMedicationsAreReturnedForNull() {
		String patientId = null;
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medications medications = dataLayer.getMedications(pId, true);
		assertEquals(0, medications.size());
	}

	@Test
	public void validateZeroMedicationsAreReturnedForEmpty() {
		String patientId = null;
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medications medications = dataLayer.getMedications(pId, true);
		assertEquals(0, medications.size());
	}

	@Test
	public void validateMedicationDataIsReturned() {
		String orderNumber = "1";
		String expectedDrugName = "AMITRIPTYLINE HCL 50 MG TAB";
		String expectedMedSoruce = "OUTPATIENT";
		String expectedMedCategory = "Rx";
		String expectedMedPharmacy = "CVS";
		Medication retrievedMedication = dataLayer.getMedicationLite(createPid(), orderNumber);
		assertEquals(expectedDrugName, retrievedMedication.getDrugName());
		assertEquals(expectedMedSoruce, retrievedMedication.getMedicationSource());
		assertEquals(expectedMedCategory, retrievedMedication.getCategory());
		assertEquals(expectedMedPharmacy, retrievedMedication.getPharmacy());
		assertNull(retrievedMedication.getMedicationDetail());
	}

	@Test
	public void validateMedicationDataIsNullForInvalidOrderNumber() {
		String orderNumber = "saf";
		Medication retrievedMedication = dataLayer.getMedicationLite(createPid(), orderNumber);
		assertNull(retrievedMedication);
	}

	@Test
	public void validateMedicationDataIsNullForInvalidPatientAndOrderNumber() {
//		String patientId = "D123401asf";
		String orderNumber = "saf";
		Medication retrievedMedication = dataLayer.getMedicationLite(createPid(), orderNumber);
		assertNull(retrievedMedication);
	}

	@Test
	public void validateMedicationDetailIsReturned() {
		String orderNumber = "1";
		String expectedDrugName = "AMITRIPTYLINE HCL 50 MG TAB";
		String expectedMedSoruce = "OUTPATIENT";
		Medication retrievedMedication = dataLayer.getMedicationDetail(createPid(), orderNumber);
		assertEquals(expectedDrugName, retrievedMedication.getDrugName());
		assertEquals(expectedMedSoruce, retrievedMedication.getMedicationSource());
		assertEquals("text", retrievedMedication.getMedicationDetail());
	}

	@Test
	public void validateMedicationDetailIsNullForInvalidOrderNumber() {
		String orderNumber = "saf";
		Medication retrievedMedication = dataLayer.getMedicationDetail(createPid(), orderNumber);
		assertNull(retrievedMedication);
	}

	@Test
	public void validateMedicationDetailIsNullForInvalidPatientAndOrderNumber() {
		String patientId = "D123401asf";
		String orderNumber = "saf";
		PatientIdentifier pId = new PatientIdentifier();
		pId.setUniqueId(patientId);
		Medication retrievedMedication = dataLayer.getMedicationDetail(pId, orderNumber);
		assertNull(retrievedMedication);
	}

}
