package com.agilex.healthcare.mobilehealthplatform.datalayer.lab;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterImpl;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.MutableDateFilter;
import com.agilex.healthcare.utility.DateHelper;

public class ChemistryLabDataLayerExcelTest {

	private PatientIdentifier createPatientIdentifier() {
		PatientIdentifier patientIdentifier = new PatientIdentifier();
		patientIdentifier.setAssigningAuthority("dfn-500");
		patientIdentifier.setUniqueId("D123401");
		return patientIdentifier;
	}

	@Test
	public void validateFirstRow() {
		PatientIdentifier pId = createPatientIdentifier();
		ChemistryLabDataLayerExcel dataLayer = new ChemistryLabDataLayerExcel();
		MutableDateFilter dateFilter = new DateFilterImpl();
		dateFilter.setStartDate(DateHelper.parseDate("1/1/2001"));
		dateFilter.setEndDate(new Date());
		LabResults labResults = dataLayer.getLabResults(pId, dateFilter);
		assertTrue(labResults.size() > 0);

		LabResult glucoseTest = null;
		for (LabResult labResult : labResults) {
			if (labResult.getAccessionNumber().contentEquals("1") && labResult.getTestId().contentEquals("5377")) {
				glucoseTest = labResult;
			}
		}

		assertNotNull(glucoseTest);
		assertEquals("1", glucoseTest.getAccessionNumber());
		assertEquals("GLUC", glucoseTest.getTestname());
		assertEquals("300", glucoseTest.getReferenceHigh());
		assertEquals("60", glucoseTest.getReferenceLow());
		assertEquals("103", glucoseTest.getValue());
		assertEquals("72", glucoseTest.getSpecimenId());
		assertEquals("SERUM", glucoseTest.getSpecimenName());
		assertEquals("C", glucoseTest.getLabType());
		assertEquals("Glucose", glucoseTest.getDisplayDescription());
	}

	@Test
	public void validateSecondRow() {
		PatientIdentifier pId = createPatientIdentifier();
		ChemistryLabDataLayerExcel dataLayer = new ChemistryLabDataLayerExcel();
		MutableDateFilter dateFilter = new DateFilterImpl();
		dateFilter.setStartDate(DateHelper.parseDate("1/1/2001"));
		dateFilter.setEndDate(new Date());
		LabResults labResults = dataLayer.getLabResults(pId, dateFilter);
		assertTrue(labResults.size() > 0);

		LabResult glucoseTest = null;
		for (LabResult labResult : labResults) {
			if (labResult.getAccessionNumber().contentEquals("2") && labResult.getTestId().contentEquals("5377")) {
				glucoseTest = labResult;
			}
		}

		assertNotNull(glucoseTest);
		assertEquals("2", glucoseTest.getAccessionNumber());
		assertEquals("GLUC", glucoseTest.getTestname());
		assertEquals("300", glucoseTest.getReferenceHigh());
		assertEquals("60", glucoseTest.getReferenceLow());
		assertEquals("84", glucoseTest.getValue());
		assertEquals("72", glucoseTest.getSpecimenId());
		assertEquals("SERUM", glucoseTest.getSpecimenName());
		assertEquals("C", glucoseTest.getLabType());

	}

	@Test
	public void validateIndicatorOnAbnormalTest() {
		String patientId = "D123401";
		String accessionNumber = "13";
		String testId = "5377";

		LabResult labResult = getLabResult(patientId, testId, accessionNumber);

		assertNotNull(labResult);
		assertEquals("L", labResult.getIndicator());
	}

	@Test
	public void validateIndicatorOnNormalTest() {
		String patientId = "D123401";
		String accessionNumber = "12";
		String testId = "5377";

		LabResult labResult = getLabResult(patientId, testId, accessionNumber);

		assertNotNull(labResult);
		assertEquals("", labResult.getIndicator());
	}

	@Test
	public void validateChemTestsMarked() {
		PatientIdentifier pId = createPatientIdentifier();
		ChemistryLabDataLayerExcel dataLayer = new ChemistryLabDataLayerExcel();
		MutableDateFilter dateFilter = new DateFilterImpl();
		dateFilter.setStartDate(DateHelper.parseDate("1/1/2001"));
		dateFilter.setEndDate(new Date());
		LabResults labResults = dataLayer.getLabResults(pId, dateFilter);
		assertTrue(labResults.size() > 0);

		for (LabResult labResult : labResults) {
			assertEquals("C", labResult.getLabType());
		}
	}

	private LabResult getLabResult(String patientId, String testId, String accessionNumber) {
		PatientIdentifier pId = createPatientIdentifier();
		ChemistryLabDataLayerExcel dataLayer = new ChemistryLabDataLayerExcel();
		MutableDateFilter dateFilter = new DateFilterImpl();
		dateFilter.setStartDate(DateHelper.parseDate("1/1/2001"));
		dateFilter.setEndDate(new Date());
		LabResults labResults = dataLayer.getLabResults(pId, dateFilter);
		assertTrue(labResults.size() > 0);

		LabResult labResult = null;
		for (LabResult possibleMatchingLabResult : labResults) {
			if (possibleMatchingLabResult.getAccessionNumber().contentEquals(accessionNumber) && possibleMatchingLabResult.getTestId().contentEquals(testId)) {
				labResult = possibleMatchingLabResult;
			}
		}
		return labResult;
	}
}
