package com.agilex.healthcare.mobilehealthplatform.datalayer.exercise;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.event.ExerciseDataLayerExcel;
import com.agilex.healthcare.mobilehealthplatform.domain.Exercise;
import com.agilex.healthcare.mobilehealthplatform.domain.Exercises;
import com.agilex.healthcare.utility.DateHelper;

public class ExcerciseDataLayerExcelTest {
	
	ExerciseDataLayer dl = new ExerciseDataLayerExcel();

	@Test
	public void getPrestagedExercises() {
		String gallowPatientId = "D123401";
		ExerciseDataLayer dl = new ExerciseDataLayerExcel();
		Exercises exercises = dl.getExercises(gallowPatientId);
		assertTrue(exercises.size() > 0);
		
		Exercise exercise01 = exercises.get(0);
		assertEquals("1", exercise01.getUniqueId());

		assertEquals(DateHelper.parseDate("1/1/2012"), exercise01.getDate());
	}
	
	@Test
	public void get0ResultsForInvalidPatient() {
		String patientId = "invalid";
		Exercises exercises = dl.getExercises(patientId);
		assertEquals(0, exercises.size());
	}
	
	@Test
	public void addAndRetrieveExercises() {
		String patientId = generateId();
		Exercises exercises = dl.getExercises(patientId);
		assertEquals(0, exercises.size());
		
		Exercise exercise = new Exercise();
		exercise.setPatientId(patientId);
		Exercise savedEx = dl.saveExercise(exercise);
		
		Exercises retrieved = dl.getExercises(patientId);
		assertEquals(1, retrieved.size());
		assertEquals(savedEx.getUniqueId(), retrieved.get(0).getUniqueId());
	}
	
	@Test
	public void retrieveWithDateFilter() {
		String patientId = generateId();
		
		Exercise exercise = new Exercise();
		exercise.setPatientId(patientId);
		exercise.setDate(DateHelper.parseDate("1/12/2012"));
		dl.saveExercise(exercise);
		
		exercise = new Exercise();
		exercise.setPatientId(patientId);
		exercise.setDate(DateHelper.parseDate("1/13/2012"));
		dl.saveExercise(exercise);
		
		exercise = new Exercise();
		exercise.setPatientId(patientId);
		exercise.setDate(DateHelper.parseDate("1/15/2012"));
		dl.saveExercise(exercise);
		
		ExerciseFilter filter = new ExerciseFilter();
		filter.setStartDate(DateHelper.parseDate("1/12/2012"));
		filter.setEndDate(DateHelper.parseDate("1/14/2012"));
		Exercises shouldBeTwo = dl.getExercises(patientId, filter);
		assertEquals(2, shouldBeTwo.size());
		
		filter = new ExerciseFilter();
		filter.setStartDate(DateHelper.parseDate("1/12/2012"));
		filter.setEndDate(DateHelper.parseDate("1/15/2012"));
		Exercises shouldBeThree = dl.getExercises(patientId, filter);
		assertEquals(3, shouldBeThree.size());
		
		filter = new ExerciseFilter();
		filter.setStartDate(DateHelper.parseDate("1/12/2012"));
		filter.setEndDate(DateHelper.parseDate("1/12/2012"));
		Exercises shouldBeOne = dl.getExercises(patientId, filter);
		assertEquals(1, shouldBeOne.size());
	}
	
	@Test 
	public void validateUpdate() {
		String patientId = generateId();
		
		Exercise exercise = new Exercise();
		exercise.setPatientId(patientId);
		exercise.setActivity("1st");
		exercise.setDate(DateHelper.parseDate("1/12/2012"));
		Exercise saved = dl.saveExercise(exercise);
		assertNotNull(saved);
		assertEquals("1st", saved.getActivity());
		
		saved.setActivity("2nd");
		Exercise updated = dl.saveExercise(saved);
		
		assertEquals("2nd", updated.getActivity());
	}
	
	@Test 
	public void validateDelete() {
		String patientId = generateId();
		
		Exercise exercise = new Exercise();
		exercise.setPatientId(patientId);
		exercise.setDate(DateHelper.parseDate("1/12/2012"));
		dl.saveExercise(exercise);
		
		Exercises shouldBeOne = dl.getExercises(patientId);
		assertEquals(1, shouldBeOne.size());
		
		dl.deleteExercise(shouldBeOne.get(0));
		
		Exercises shouldBeEmpty = dl.getExercises(patientId);
		assertEquals(0, shouldBeEmpty.size());
	}
	
	private String generateId() {
		return UUID.randomUUID().toString();
	}
}
