package com.agilex.healthcare.mobilehealthplatform.datalayer.diet;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;
import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.DietEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.DietEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.utility.DateHelper;

public class DietDataLayerExcelTest {

	DietDataLayer dl = new DietDataLayerExcel();

	@Test
	public void getPrestagedDietEntries() {
		String patientId = "PATID26";
		DietDataLayer dl = new DietDataLayerExcel();
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertTrue(dietEntries.size() > 0);

		DietEntry dietEntry01 = dietEntries.get(0);
		assertEquals("1", dietEntry01.getUniqueId());

		assertEquals(DateHelper.parseDate("12/1/2011"), dietEntry01.getDate());
	}

	@Test
	public void get0ResultsForInvalidPatient() {
		String patientId = "invalid";
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertEquals(0, dietEntries.size());
	}

	@Test
	public void addAndRetrieveDietEntries() {
		String patientId = generateId();
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertEquals(0, dietEntries.size());

		DietEntry dietEntry = new DietEntry();
		dietEntry.setPatientId(patientId);
		DietEntry savedDietEntry = dl.saveDietEntry(dietEntry);

		DietEntries retrieved = dl.getDietEntries(patientId);
		assertEquals(1, retrieved.size());
		assertEquals(savedDietEntry.getUniqueId(), retrieved.get(0).getUniqueId());
	}

	@Test
	public void retrieveWithDateFilter() {
		String patientId = generateId();

		DietEntry dietEntry = new DietEntry();
		dietEntry.setPatientId(patientId);
		dietEntry.setEntryDate(DateHelper.parseDate("1/12/2012"));
		dl.saveDietEntry(dietEntry);

		dietEntry = new DietEntry();
		dietEntry.setPatientId(patientId);
		dietEntry.setEntryDate(DateHelper.parseDate("1/13/2012"));
		dl.saveDietEntry(dietEntry);

		dietEntry = new DietEntry();
		dietEntry.setPatientId(patientId);
		dietEntry.setEntryDate(DateHelper.parseDate("1/15/2012"));
		dl.saveDietEntry(dietEntry);

		Date startDate = DateHelper.parseDate("1/12/2012");
		Date endDate = DateHelper.parseDate("1/14/2012");
		DateFilter filter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		DietEntries shouldBeTwo = dl.getDietEntries(patientId, filter);
		assertEquals(2, shouldBeTwo.size());
		
		startDate = DateHelper.parseDate("1/12/2012");
		endDate = DateHelper.parseDate("1/15/2012");
		filter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		DietEntries shouldBeThree = dl.getDietEntries(patientId, filter);
		assertEquals(3, shouldBeThree.size());
		
		startDate = DateHelper.parseDate("1/12/2012");
		endDate = DateHelper.parseDate("1/12/2012");
		filter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		DietEntries shouldBeOne = dl.getDietEntries(patientId, filter);
		assertEquals(1, shouldBeOne.size());
	}

	@Test
	public void validateUpdate() {
		String patientId = generateId();

		DietEntry dietEntry = new DietEntry();
		dietEntry.setPatientId(patientId);
		dietEntry.setFat("1");
		dietEntry.setEntryDate(DateHelper.parseDate("1/12/2012"));
		DietEntry saved = dl.saveDietEntry(dietEntry);
		assertNotNull(saved);
		assertEquals("1", saved.getFat());

		saved.setFat("2");
		DietEntry updated = dl.saveDietEntry(saved);

		assertEquals("2", updated.getFat());
	}

	@Test
	public void validateDelete() {
		String patientId = generateId();

		DietEntry dietEntry = new DietEntry();
		dietEntry.setPatientId(patientId);
		dietEntry.setEntryDate(DateHelper.parseDate("1/12/2012"));
		dl.saveDietEntry(dietEntry);

		DietEntries shouldBeOne = dl.getDietEntries(patientId);
		assertEquals(1, shouldBeOne.size());

		dl.deleteDietEntry(shouldBeOne.get(0));

		DietEntries shouldBeEmpty = dl.getDietEntries(patientId);
		assertEquals(0, shouldBeEmpty.size());
	}

	private String generateId() {
		return UUID.randomUUID().toString();
	}
	
	@Test
	public void testMealType() {
		String patientId = "PATID26";
		DietDataLayer dl = new DietDataLayerExcel();
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertTrue(dietEntries.size() > 0);

		DietEntry dietEntry01 = dietEntries.get(0);
		assertEquals("Breakfast", dietEntry01.getMealType().toString());

		DietEntry dietEntry02 = dietEntries.get(1);
		assertEquals("Lunch", dietEntry02.getMealType().toString());	
	}

	@Test
	public void testEntryDate() {
		String patientId = "PATID26";
		DietDataLayer dl = new DietDataLayerExcel();
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertTrue(dietEntries.size() > 0);

		DietEntry dietEntry01 = dietEntries.get(0);
		assertEquals(DateHelper.parseDate("12/1/2011"), dietEntry01.getDate());

		DietEntry dietEntry02 = dietEntries.get(1);
		assertEquals(DateHelper.parseDate("12/1/2011"), dietEntry02.getDate());							
	}

	@Test
	public void testFat() {
		String patientId = "PATID26";
		DietDataLayer dl = new DietDataLayerExcel();
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertTrue(dietEntries.size() > 0);

		DietEntry dietEntry01 = dietEntries.get(0);
		assertEquals("3", dietEntry01.getFat());

		DietEntry dietEntry02 = dietEntries.get(1);
		assertEquals("1", dietEntry02.getFat());			
	}

	@Test
	public void testProtein() {
		String patientId = "PATID26";
		DietDataLayer dl = new DietDataLayerExcel();
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertTrue(dietEntries.size() > 0);

		DietEntry dietEntry01 = dietEntries.get(0);
		assertEquals("8", dietEntry01.getProtein());

		DietEntry dietEntry02 = dietEntries.get(1);
		assertEquals("4", dietEntry02.getProtein());					
	}

	@Test
	public void testCalories() {
		String patientId = "PATID26";
		DietDataLayer dl = new DietDataLayerExcel();
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertTrue(dietEntries.size() > 0);

		DietEntry dietEntry01 = dietEntries.get(0);
		assertEquals("250", dietEntry01.getCalories());

		DietEntry dietEntry02 = dietEntries.get(1);
		assertEquals("100", dietEntry02.getCalories());					
	}

	@Test
	public void testCarbs() {
		String patientId = "PATID26";
		DietDataLayer dl = new DietDataLayerExcel();
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertTrue(dietEntries.size() > 0);

		DietEntry dietEntry01 = dietEntries.get(0);
		assertEquals("15", dietEntry01.getCarbs());

		DietEntry dietEntry02 = dietEntries.get(1);
		assertEquals("9", dietEntry02.getCarbs());							
	}

	@Test
	public void testNotes() {
		String patientId = "PATID26";
		DietDataLayer dl = new DietDataLayerExcel();
		DietEntries dietEntries = dl.getDietEntries(patientId);
		assertTrue(dietEntries.size() > 0);

		DietEntry dietEntry01 = dietEntries.get(0);
		assertEquals("An Omlet", dietEntry01.getNotes());

		DietEntry dietEntry02 = dietEntries.get(1);
		assertEquals("Potato Chips", dietEntry02.getNotes());							
	}
}
