package com.agilex.healthcare.mobilehealthplatform.datalayer.contactlog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;
import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.ContactLog;
import com.agilex.healthcare.mobilehealthplatform.domain.ContactLogs;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.utility.DateHelper;

public class ContactLogDataLayerMockTest {

	ContactLogDataLayer dl = new ContactLogDataLayerMock();

	private void init(String patientId) {
		ContactLog log = new ContactLog();
		log.setActivity("act-01");
		log.setContact("con-01");
		log.setNote("not-01");
		log.setTask("tas-01");
		log.setLogTime(DateHelper.parseDate("3/1/2012"));
		log.setPatientId(patientId);
		dl.saveContactLog(log);

		log = new ContactLog();
		log.setActivity("act-02");
		log.setContact("con-02");
		log.setNote("not-02");
		log.setTask("tas-02");
		log.setLogTime(DateHelper.parseDate("3/2/2012"));
		log.setPatientId(patientId);
		dl.saveContactLog(log);

		log = new ContactLog();
		log.setActivity("act-03");
		log.setContact("con-03");
		log.setNote("not-03");
		log.setTask("tas-03");
		log.setLogTime(DateHelper.parseDate("3/3/2012"));
		log.setPatientId(patientId);
		dl.saveContactLog(log);

		log = new ContactLog();
		log.setActivity("act-04");
		log.setContact("con-04");
		log.setNote("not-04");
		log.setTask("tas-04");
		log.setLogTime(DateHelper.parseDate("3/4/2012"));
		log.setPatientId(patientId);
		dl.saveContactLog(log);

		log = new ContactLog();
		log.setActivity("act-05");
		log.setContact("con-05");
		log.setNote("not-05");
		log.setTask("tas-05");
		log.setPatientId(patientId);
		log.setLogTime(DateHelper.parseDate("3/5/2012"));
		dl.saveContactLog(log);
	}

	@Test
	public void verifygetAllContactLogsById() {
		String patientId = "01";
		init("01");
		ContactLogs contactLogs = dl.getContactLogs(patientId);
		assertEquals(5, contactLogs.size());
	}

	@Test
	public void verfiyGetByContactLogId() {
		String patientId = "02";
		ContactLog log = new ContactLog();
		log.setActivity("act-02");
		log.setContact("con-02");
		log.setNote("not-02");
		log.setTask("tas-02");
		log.setPatientId(patientId);
		ContactLog savedContactLog = dl.saveContactLog(log);
		assertNotNull(savedContactLog);
		assertNotNull(savedContactLog.getId());
		assertEquals(patientId, savedContactLog.getPatientId());

		ContactLog retrievedLog = dl.getContactLog(patientId, savedContactLog.getId());
		assertEquals(savedContactLog.getId(), retrievedLog.getId());
		assertEquals(log.getActivity(), retrievedLog.getActivity());
		assertEquals(patientId, retrievedLog.getPatientId());
	}

	@Test
	public void verifySaveContactLogWithExistingPatientId() {
		String patientId = "03";
		ContactLog logToSave = new ContactLog();
		logToSave = new ContactLog();

		logToSave.setActivity("Request to save log");
		logToSave.setContact("Pharmacist");
		logToSave.setLogTime(new Date());
		logToSave.setNote("This should properly save");
		logToSave.setPatientId(patientId);
		logToSave.setTask("Properly Save the Contact Log");

		ContactLogDataLayer dl = new ContactLogDataLayerMock();
		ContactLog savedContactLog = dl.saveContactLog(logToSave);

		assertNotNull(savedContactLog);
		assertNotNull(savedContactLog.getId());
		assertEquals("Request to save log", savedContactLog.getActivity());
		assertEquals("Pharmacist", savedContactLog.getContact());
		assertEquals(patientId, savedContactLog.getPatientId());
	}

	@Test
	public void verifySaveContactLogWithNewPatientId() {
		ContactLog logToSave = new ContactLog();
		logToSave = new ContactLog();

		logToSave.setActivity("Request to save log");
		logToSave.setContact("Pharmacist");
		logToSave.setLogTime(new Date());
		logToSave.setNote("This should properly save");
		logToSave.setPatientId("uniqueId");
		logToSave.setTask("Properly Save the Contact Log");

		ContactLogDataLayer dl = new ContactLogDataLayerMock();
		ContactLog savedContactLog = dl.saveContactLog(logToSave);

		assertNotNull(savedContactLog.getId());
		assertEquals("uniqueId", logToSave.getPatientId());
		assertEquals("Request to save log", logToSave.getActivity());
		assertEquals("Pharmacist", logToSave.getContact());

		ContactLog retrieved = dl.getContactLog("uniqueId", savedContactLog.getId());
		assertEquals("uniqueId", retrieved.getPatientId());
		assertEquals("Request to save log", retrieved.getActivity());
		assertEquals("Pharmacist", retrieved.getContact());
	}

	@Test
	public void verifyFilterOnLowerBoundary() {
		String patientId = "04";
		init(patientId);
		DateFilter filter = DateFilterFactory.createFilterFromDate("3/1/2012", "3/1/2012");
		ContactLogDataLayer dl = new ContactLogDataLayerMock();

		ContactLogs shouldBeOne = dl.getContactLogs(patientId, filter);
		assertEquals(1, shouldBeOne.size());
	}

	@Test
	public void verifyFilterOnSingleDate() {
		String patientId = UUID.randomUUID().toString();
		init(patientId);
		DateFilter filter = DateFilterFactory.createFilterFromDate("3/1/2012");
		ContactLogDataLayer dl = new ContactLogDataLayerMock();

		ContactLogs shouldBeOne = dl.getContactLogs(patientId, filter);
		assertEquals(1, shouldBeOne.size());
	}

	@Test
	public void verifyFilterOnLowerBoundaryWithNoUpper() {
		String patientId = "05";
		init(patientId);
		DateFilter filter = DateFilterFactory.createFilterFromDate("3/1/2012", null);
		ContactLogDataLayer dl = new ContactLogDataLayerMock();

		ContactLogs shouldBeMany = dl.getContactLogs(patientId, filter);
		assertTrue(shouldBeMany.size() >= 5);
	}

	@Test
	public void verifyFilterOnUpperBoundaryWithNoLower() {
		String patientId = "06";
		init(patientId);
		DateFilter filter = DateFilterFactory.createFilterFromDate(null, "3/5/2012");
		ContactLogDataLayer dl = new ContactLogDataLayerMock();

		ContactLogs shouldBeMany = dl.getContactLogs(patientId, filter);
		assertTrue(shouldBeMany.size() >= 5);
	}

	@Test
	public void verifyFilterOnUpperBoundary() {
		String patientId = "07";
		init(patientId);
		DateFilter filter = DateFilterFactory.createFilterFromDate("3/5/2012", "3/5/2012");
		ContactLogDataLayer dl = new ContactLogDataLayerMock();

		ContactLogs shouldBeOne = dl.getContactLogs(patientId, filter);
		assertEquals(1, shouldBeOne.size());
	}

	@Test
	public void verifyFilterOnUpperBoundaryWithSingleDate() {
		String patientId = UUID.randomUUID().toString();
		init(patientId);
		DateFilter filter = DateFilterFactory.createFilterFromDate("3/5/2012");
		ContactLogDataLayer dl = new ContactLogDataLayerMock();

		ContactLogs shouldBeOne = dl.getContactLogs(patientId, filter);
		assertEquals(1, shouldBeOne.size());
	}

	@Test
	public void verifyDeleteContactLog() {
		ContactLog logToSave = new ContactLog();
		logToSave = new ContactLog();
		String patientId = "08";
		logToSave.setActivity("Request to save log");
		logToSave.setContact("Pharmacist");
		logToSave.setLogTime(new Date());
		logToSave.setNote("This should properly save");
		logToSave.setPatientId(patientId);
		logToSave.setTask("Properly Save the Contact Log");
		ContactLog savedLog = dl.saveContactLog(logToSave);

		assertNotNull(savedLog.getId());

		ContactLog logToDelete = new ContactLog();
		logToDelete.setId(savedLog.getId());
		logToDelete.setPatientId(patientId);
		dl.deleteContactLog(logToDelete);

		ContactLog shouldBeNull = dl.getContactLog(patientId, savedLog.getId());
		assertNull(shouldBeNull);
	}
}
